<?php

namespace AvengersMG\MGCms2019\App\Http\Controllers\Frontend;

use \Exception;
use \DateTime;
use \Throwable;
use App;
use AvengersMG\MGCms2019\App\Cms\Accommodations\Rooms\RoomInterface;
use AvengersMG\MGCms2019\App\Cms\Apis\Adapters\ApiAdapter;
use AvengersMG\MGCms2019\App\Cms\Apis\Calendar;
use AvengersMG\MGCms2019\App\Cms\Apis\PriceTravel\PriceTravelApi;
use AvengersMG\MGCms2019\App\Cms\Availabilities\Availability;
use AvengersMG\MGCms2019\App\Cms\Availabilities\Repositories\AvailabilityInterface;
use AvengersMG\MGCms2019\App\Cms\Components\Specials\SpecialInterface;
use AvengersMG\MGCms2019\App\Cms\Components\Tours\Tour;
use AvengersMG\MGCms2019\App\Cms\Forms\FormsInterface;
use AvengersMG\MGCms2019\App\Cms\Pages\Page;
use AvengersMG\MGCms2019\App\Cms\Pages\Repositories\PageInterface;
use AvengersMG\MGCms2019\App\Cms\Reviews\Repositories\ReviewsRepository;
use AvengersMG\MGCms2019\App\Cms\Sites\Site;
use AvengersMG\MGCms2019\App\Http\Controllers\Controller;
use Carbon\Carbon;
use GuzzleHttp\Client;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Input;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class PageController extends Controller
{
    private $pageRepository;
    private $roomRepository;
    private $availableRepository;
    private $reviewsRepository;
    private $specialsRepository;
    private $formsRepository;
    /** @var string  Fomato en el que las fechas vienen desde el frontend */
    private $defaultDateFormat = 'm/d/Y';

    public function __construct(
        PageInterface $pageInterface,
        RoomInterface $roomInterface,
        AvailabilityInterface $availableInterface,
        ReviewsRepository $reviewsRepository,
        SpecialInterface $specialInterface,
        FormsInterface $formsInterface
    ) {
        $this->pageRepository = $pageInterface;
        $this->roomRepository = $roomInterface;
        $this->availableRepository = $availableInterface;
        $this->reviewsRepository = $reviewsRepository;
        $this->specialsRepository = $specialInterface;
        $this->formsRepository = $formsInterface;
    }

    public function index(Request $request)
    {
        /** @var mixed[] Variables de retorno para la vista */
        $returnData = [];

        $uri = $request->path();
        $uri = ($uri == "/") ? $uri : '/' . $uri;
        $uri = explode('?', $uri);

        //lenguaje
        $lang = $this->lang($uri[0]);

        /* Establecer variables de reservación */
        if ($request->has(['rooms', 'arrivalDate', 'departureDate'])) {
            $validator = $this->createPriceRequestValidator($request);

            if ($validator->passes()) {
                $returnData['room_search_request'] = $request->all();
                /* Pasar los datos en crudo */
            }
        }

        /* Si no está establecido, poner datos predeterminados */
        if (!isset($returnData['room_search_request'])) {
            /* Si no hay datos establecidos, poner datos predeterminados */
            $returnData['room_search_request'] = $this->createDefaultSearchValues($lang);
        }

        $site = Site::current();

        $navigations = $this->pageRepository->getNavigation($site);
        $page = $this->pageRepository->filterByPermalink($uri[0], $site);

        /* Remover la página si el estado es "hidden" */
        if (is_a($page, Page::class)) {
            if ($page->status == "hidden") {
                /* Verificar si la página tiene un especial */
                if ($page->special()->count() > 0) {
                    $roomPricesPage = $site->getRoomPricesPage();
                    $pageIndex = $site->pages()->withDepth()->having('depth', '=', 0)->first();

                    $redirectPage = !empty($roomPricesPage)
                        /* Solo redirigir si existe la página */
                        ? $roomPricesPage->translate($lang)
                        /* No existe la página de reservaciones. Redirigir al inicio en el idioma solicitado */
                        : $redirectPage = $pageIndex->translate($lang);

                    /* Abortar proceso y redirigir */
                    return redirect($redirectPage->perman_link);
                } else {
                    /* la página no tiene especial, destruir */
                    $page = null;
                }
            }
        }


        /* No usar verificación sólo cuando el ambiente es local */
        $use_verify_peer = (App::environment('local')) ? false : true;

        /* Establecer las opciones para el contexto de flujo */
        $arrContextOptions = [
            "ssl" => [
                "verify_peer" => $use_verify_peer,
                "verify_peer_name" => $use_verify_peer,
            ],
        ];

        /* Obtener fechas no disponibles */
        $blackoutDates = $this->availableRepository->getTotalBlackoutDates(null, null, $this->defaultDateFormat, $site);

        if ($page) {
            /* Cambiado a usar permalink para evitar usar ID interna */
            if (in_array($page->perman_link, ['/', "/{$lang}"])) {
                if ($page->locale == 'en') {
                    $page->feedUrl = config("constants.site{$site->id}.BlogUrl.en");
                } else {
                    $page->feedUrl =  config("constants.site{$site->id}.BlogUrl.es");
                }
                ini_set('allow_url_fopen', 1);
                if (ini_get('allow_url_fopen')) {
                    try {
                        $feed = file_get_contents($page->feedUrl, false, stream_context_create($arrContextOptions));
                        $feed = new \SimpleXmlElement($feed);
                        $page->feed = $feed->channel->item;
                    } catch (Exception $e) {
                        $page->feed = [];
                    }
                } else {
                    $page->feed = [];
                }
            }

            /* Cargar relaciones adicionales */
            $this->loadRelationsToModel($page);

            $descendants = $page->descendants()->with('translations')->with('mediafiles.type')->whereHas('translations', function ($query) {
                $query->where('status', 'published');
            })->defaultOrder()->get()->toTree();
            $ancestors = $page->ancestors()->with(['translations'])->whereHas('translations', function ($query) {
                $query->where('status', 'published');
            })->defaultOrder()->get()->toTree();
            $mapPages = $this->pageRepository->toTrees($site)->where('status', 'published');
            $reviews = $this->reviewsRepository->getReviews($lang);
            /** 2020-06-12: Añadida una revisión redundante de publicación específica de la traducción porque getActiveSpecialsBySite() trae la página con que al menos UNA traducción esté en "published" */
            $specials = $this->pageRepository->getActiveSpecialsBySite($site)->translatedIn($lang)->whereTranslation('status', 'published', $lang)->with(['mediafiles.type'])->defaultOrder()->get();
            $forms = $this->formsRepository->list();

            /* Añadir variables de página al retorno */
            $returnData = array_merge(
                $returnData,
                compact(
                    'page',
                    'navigations',
                    'descendants',
                    'ancestors',
                    'lang',
                    'blackoutDates',
                    'mapPages',
                    'reviews',
                    'specials',
                    'forms'
                )
            );

            /**
             * Si se hace una peticion Ajax retorna solo los datos de la pagina.
             * Asegurar encabezados JSON
             */
            if ($request->wantsJson()) {
                return array_merge(
                    [],
                    compact(
                        'page',
                        'descendants',
                        'ancestors',
                        'lang',
                        'forms'
                    )
                );
            }

            return view('layouts.' . $page->layout, $returnData);
        } else {
            /* Ruta actual para usos futuros */
            $current_route = Route::getCurrentRoute();

            /* Buscar rutas alternativas en caso que no exista la página */
            $matching_routes = collect(Route::getRoutes()->get())
                ->filter(function ($value, $key) use ($request, $current_route) {
                    return ($value->matches($request)
                        && ($value->uri != $current_route->uri));
                });

            /* Ejecutar el primer resultado en caso que lo haya */
            if ($matching_routes->isNotEmpty()) {
                return $matching_routes->first()->bind($request)->run();
            }

            /* Sin resultados */
            return abort(404);
        }
    }

    public function lang($uri)
    {
        $uri = explode('/', $uri);
        ($uri[1] == 'es') ? app()->setLocale($uri[1]) : '';
        return $lang = app()->getLocale();
    }

    public function isSpecial($special)
    {
        if ($special) {
            if ($special->end_date <= date('d-m-Y')) {
                return false;
            } else {
                return true;
            }
        } else {
            return true;
        }
    }


    public function tour(Request $request)
    {
        $page = Page::find($request->page);

        /* Cargar relaciones adicionales */
        $this->loadRelationsToModel($page);

        $site = $page->site;
        $lang = $request->lang;
        $navigations = Page::where('site_id', $site->id)->whereTranslation('navigation', 1)->whereTranslation('status', 'published')->get()->toTree();
        $static = true;

        return view('pages.tour-form', compact('page', 'site', 'lang', 'navigations', 'static'));
    }

    /**
     * Recupera la información de precios y disponibilida de habitaciones para vista
     *
     * @param  Request  $request  Los datos que el usuario solicitó, incluida la localzación
     * @return Response           Respuesta a mostrar
     */
    public function roomPrices(Request $request)
    {
        if ($request->has(['rooms', 'arrivalDate', 'departureDate'])) {
            /*
                Validar el request
             */
            $validator = $this->createPriceRequestValidator($request);

            /* Revisar si es correcto */
            if ($validator->fails()) {
                return redirect()->back()->withErrors($validator)->withInput();
            }

            /* Validador pasó */

            $rooms = $request->rooms;

            /** @var string Fecha de arribo en formato mm/dd/yyyy */
            $arrivalDate = $request->input('arrivalDate');

            /** @var string Fecha de salida en formato mm/dd/yyyy */
            $departureDate = $request->input('departureDate');

            /** @var string Lenguaje solicitado (Ej. 'en', 'es'...) */
            $language = $request->input('language');

            /** @var string Moneda solicitada (Ej. 'MXN', 'USD'...) */
            $currency = $request->currency;
        } else {
            /* Parsear y establecer lenguaje */
            $uri = \request()->path();
            $uri = ($uri == "/") ? $uri : '/' . $uri;
            $uri = explode('?', $uri);

            /* Si no hay request, poner datos predeterminados en el ambiente */
            extract(
                $this->createDefaultSearchValues(
                    $this->lang($uri[0])/* Establecer el lenguaje y mandarlo */
                )
            );
        }

        /* Cambiar por el lenguaje deseado */
        app()->setLocale($language);

        /* Resolver el adaptador de API */
        $api = App::make(ApiAdapter::class);

        /* Configurar API */
        $api
            ->setRoomsData($rooms)
            ->setLanguage($language)
            ->setCurrency($currency)
            ->setArrivalDate($arrivalDate, $this->defaultDateFormat)
            ->setDepartureDate($departureDate, $this->defaultDateFormat);

        /* Obtener habitaciones y precios */
        $roomsData = $api->getAvailabilityDataForView();

        /* Responder con la información de la API, si es que es JSON */
        if ($request->has('apiId')) {
            /* Disponibilidad de una sola habitación */
            return response()->json([
                'roomData' => $roomsData['rooms']->where('api_id', $request->apiId)->first(),
                'betweenDates' => $roomsData['betweenDates'],
            ]);
        } else {
            /* Disponibilidad de todas las habitaciones */
            return response()->json($roomsData);
        }
    }

    /**
     * Función para retornar las fechas que no están disponibles en todo el hotel (sitio actual)
     * @param  Request $request Los datos que el usuario solicitó, incluida la localzación
     * @return Response         Respuesta a mostrar
     */
    public function blackoutDates(Request $request)
    {
        /* Espacio para la información de retorno */
        $retorno;

        /* Establecer sitio para usos futuros */
        $site = Site::current();

        /* Validar la información */
        $request->validate([
            'startDate' => 'nullable|date',
            'endDate' => 'nullable|date|after:startDate',
            'dateFormat' => 'nullable|string'
        ]);

        try {
            $retorno = $this->availableRepository->getTotalBlackoutDates($request->startDate, $request->endDate, $request->dateFormat, $site);
        } catch (Exception $e) {
            return response()->json(null, 500); /* Error en servidor */
        } catch (Throwable $t) {
            return response()->json(null, 422); /* Petición no ejecutable */
        }

        /* Validación exitosa. Continuar */
        return response()->json(
            $retorno
        );
    }

    /**
     * Método público para la creación de datos de sitemap
     *
     * @param  Request       $request  Los datos que el usuario solicitó
     * @param  string|null   $lang     Identificador de localización
     * @return Response                Respuesta a mostrar, en XML
     */
    public function renderSitemap(Request $request, $lang = null)
    {
        $xmlData = [];

        /* Apéndice de 's' para "HTTP", en caso de seguro */
        $secure = ($request->secure()) ? 's' : '';

        if (!is_null($lang)) {
            /*
              Si hay lenguaje, buscar todas las páginas de ese lenguaje
              que permitan ser indexadas y cuya traducción esté publicada
            */
            Page::translatedIn($lang)->whereTranslation("status", "published", $lang)->where('index', 1)->with(['site', 'translations'])->get()->each(function ($item, $index) use (&$xmlData, $secure, $lang) {
                $permalink = $item->{"perman_link:{$lang}"};
                $xmlData = array_merge($xmlData, [
                    '<url>',
                    "<loc>http{$secure}://{$item->site->domain}{$permalink}</loc>",
                    '</url>',
                ]);
            });

            /* Envolver en etiquetas de conjuntos de URL */
            $xmlData = array_merge(
                [
                    '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">',
                ],
                $xmlData,
                [
                    '</urlset>',
                ]
            );
        } else {
            /* Obtener el sitio actual */
            $site = Site::current();

            /* Ciclar los idiomas */
            collect(app()->make('translatable.locales')->all())->each(function ($item, $index) use (&$xmlData, $secure, $site, $lang) {
                $xmlData = array_merge($xmlData, [
                    '<sitemap>',
                    "<loc>http{$secure}://{$site->domain}/sitemap_{$item}.xml</loc>",
                    '</sitemap>',
                ]);
            });

            /* Envolver en etiquetas de índices */
            $xmlData = array_merge(
                [
                    '<sitemapindex xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">',
                ],
                $xmlData,
                [
                    '</sitemapindex>',
                ]
            );
        }

        /* Añadir cabecera */
        $xmlData = array_merge(
            [
                '<?xml version="1.0" encoding="UTF-8"?>',
            ],
            $xmlData
        );

        /* Parsear respuesta */
        return response(
            implode($xmlData),
            200,
            [
                'Content-Type' => 'application/xml'
            ]
        );
    }

    /**
     * Método protegido para cargar relaciones al modelo
     *
     * El modelo original será modificado
     *
     * @param  Page   $page La página en donde deben ser cargados.
     * @return void
     */
    protected function loadRelationsToModel(Page &$page)
    {
        $page
            ->loadMissing([
                'cards' => function ($query) {
                    $query->orderBy('priority', 'asc');
                },
                'galleries.items' => function ($query) {
                    $query->orderBy('priority', 'asc');
                }
            ])
            ->loadMissing([
                'cards.translations', 'mediafiles.type', 'room.amenities', 'room.mediafiles', 'special', 'restaurant', 'descendants.restaurant', 'site', 'tour','children.mediafiles.type', 'descendants.mediafiles.type', 'ancestors.translations', 'translations', 'galleries.items.mediafile', 'PageRelations.page.mediafiles.type',
            ]);
    }

    /**
     * Método protegido para el cálculo de reservaciones y retorno de
     * variables necesarias para la vista
     * @param string[]  $requestData  Datos filtrados que vienen de la vista
     * @return mixed[]                Variables necesarias para reservaciones
     */
    protected function getReservationRequestDetails(array $requestData)
    {
        /* INICIA: NUEVA IMPLEMENTACIÓN DE APIs DESACOPLADAS */

        /* Necesario resolver aquí y no en el constructor */
        $api = App::make(ApiAdapter::class);

        /* Establecer datos necesarios */
        $api
            ->setRoomsData($requestData['rooms'])
            ->setLanguage($requestData['language'])
            ->setCurrency($requestData['currency'])
            ->setArrivalDate($requestData['arrivalDate'], $this->defaultDateFormat)
            ->setDepartureDate($requestData['departureDate'], $this->defaultDateFormat);

        /* Obtener habitaciones con información de disponibilidad */
        return $api->getAvailabilityDataForView();

        /* TERMINA: NUEVA IMPLEMENTACIÓN DE APIs DESACOPLADAS */
    }

    /**
     * Método protegido para retornar un validador de solicitud de precios
     * y disponibilidad para habitación
     * @param  Request   $request   La solicitud a revisar
     * @return Validator            El validador
     */
    protected function createPriceRequestValidator(Request $request)
    {
        return Validator::make($request->all(), [
            'rooms' => 'required|array',
            'rooms.*.adults' => 'required|integer|between:1,10',
            'rooms.*.childAges' => 'sometimes|array',
            'rooms.*.childAges.*' => 'integer|between:0,12',
            'arrivalDate' => [
                'required',
                'date',
                'after:' . date($this->defaultDateFormat, time())
            ],
            'departureDate' => 'required|date|after:arrivalDate',
            'language' => [
                'required',
                Rule::in(app()->make('translatable.locales')->all())
            ],
            'currency' => [
                'required',
                Rule::in(['MXN', 'USD'])
            ],
            'apiId' => [
                'sometimes',
                'required',
                'exists:room_api_codes,api_code'
            ]
        ]);
    }

    /**
     * Método protegido para retornar datos para una búsqueda predeterminada
     * @param string        $language   Identificador de localización solicitada
     *                                  (Ej. 'en', 'es'...)
     * @param string|null   $dateFormat Formato en el que las fechas estarán
     * @return mixed[]              Arreglo de datos para búsqueda
     */
    protected function createDefaultSearchValues($language, $dateFormat = null)
    {
        /* Usar el formato predeterminado de este controlador */
        if (is_null($dateFormat)) {
            $dateFormat = $this->defaultDateFormat;  /* mm/dd/yyyy */
        }

        return [
            'rooms' => [
                [
                    'adults' => 2,
                ]
            ],
            'arrivalDate' => date($dateFormat, time()),
            'departureDate' => date($dateFormat, strtotime('+2 day')),
            'language' => $language,
            'currency' => ($language == 'es' ? 'MXN' : 'USD'),
        ];
    }

    /**
     * Metodo publico para responder por medio de un API
     */
    public function responseWithJson($request)
    {
        return $this->index($request);
    }

    /**
     * Metodo publico para traer unicamente la navegacion del sitio.
     */
    public function getNavigation()
    {
        return $this->pageRepository->getNavigation(Site::current());
    }
}
