<?php

namespace AvengersMG\MGCms2019\App\Http\Controllers\Frontend;

use \Exception;
use \DateTime;
use \Throwable;
use App;
use AvengersMG\MGCms2019\App\Cms\Accommodations\Rooms\RoomInterface;
use AvengersMG\MGCms2019\App\Cms\Apis\Adapters\ApiAdapter;
use AvengersMG\MGCms2019\App\Cms\Apis\Calendar;
use AvengersMG\MGCms2019\App\Cms\Apis\PriceTravel\PriceTravelApi;
use AvengersMG\MGCms2019\App\Cms\Availabilities\Availability;
use AvengersMG\MGCms2019\App\Cms\Availabilities\Repositories\AvailabilityInterface;
use AvengersMG\MGCms2019\App\Cms\Components\Specials\SpecialInterface;
use AvengersMG\MGCms2019\App\Cms\Components\Tours\Tour;
use AvengersMG\MGCms2019\App\Cms\Forms\FormsInterface;
use AvengersMG\MGCms2019\App\Cms\Pages\Page;
use AvengersMG\MGCms2019\App\Cms\Pages\Repositories\PageInterface;
use AvengersMG\MGCms2019\App\Cms\Reviews\Repositories\ReviewsRepository;
use AvengersMG\MGCms2019\App\Cms\Sites\Site;
use AvengersMG\MGCms2019\App\Http\Controllers\Controller;
use Carbon\Carbon;
use GuzzleHttp\Client;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Input;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;

class ApiController extends Controller
{
    private $pageRepository;
    private $roomRepository;
    private $availableRepository;
    private $reviewsRepository;
    private $specialsRepository;
    private $formsRepository;
    /** @var string  Fomato en el que las fechas vienen desde el frontend */
    private $defaultDateFormat = 'm/d/Y';

    public function __construct(
        PageInterface $pageInterface,
        RoomInterface $roomInterface,
        AvailabilityInterface $availableInterface,
        ReviewsRepository $reviewsRepository,
        SpecialInterface $specialInterface,
        FormsInterface $formsInterface
    ) {
        $this->pageRepository = $pageInterface;
        $this->roomRepository = $roomInterface;
        $this->availableRepository = $availableInterface;
        $this->reviewsRepository = $reviewsRepository;
        $this->specialsRepository = $specialInterface;
        $this->formsRepository = $formsInterface;
    }

    public function lang($uri)
    {
        $uri = explode('/', $uri);
        ($uri[1] == 'es') ? app()->setLocale($uri[1]) : '';
        return $lang = app()->getLocale();
    }

    /**
     * Recupera la información de precios y disponibilida de habitaciones para vista
     *
     * @param  Request  $request  Los datos que el usuario solicitó, incluida la localzación
     * @return Response           Respuesta a mostrar
     */
    public function roomPrices(Request $request)
    {
        if ($request->has(['rooms', 'arrivalDate', 'departureDate'])) {
            /*
                Validar el request
             */
            $validator = $this->createPriceRequestValidator($request);

            /* Revisar si es correcto */
            if ($validator->fails()) {
                return redirect()->back()->withErrors($validator)->withInput();
            }

            /* Validador pasó */

            $rooms = $request->rooms;

            /** @var string Fecha de arribo en formato mm/dd/yyyy */
            $arrivalDate = $request->input('arrivalDate');

            /** @var string Fecha de salida en formato mm/dd/yyyy */
            $departureDate = $request->input('departureDate');

            /** @var string Lenguaje solicitado (Ej. 'en', 'es'...) */
            $language = $request->input('language');

            /** @var string Moneda solicitada (Ej. 'MXN', 'USD'...) */
            $currency = $request->currency;
        } else {
            /* Parsear y establecer lenguaje */
            $uri = \request()->path();
            $uri = ($uri == "/") ? $uri : '/' . $uri;
            $uri = explode('?', $uri);

            /* Si no hay request, poner datos predeterminados en el ambiente */
            extract(
                  $this->createDefaultSearchValues(
                      $this->lang($uri[0])/* Establecer el lenguaje y mandarlo */
                  )
            );
        }

        /* Cambiar por el lenguaje deseado */
        app()->setLocale($language);

        /* Resolver el adaptador de API */
        $api = App::make(ApiAdapter::class);

        /* Configurar API */
        $api
            ->setRoomsData($rooms)
            ->setLanguage($language)
            ->setCurrency($currency)
            ->setArrivalDate($arrivalDate, $this->defaultDateFormat)
            ->setDepartureDate($departureDate, $this->defaultDateFormat);

        /* Obtener habitaciones y precios */
        $roomsData = $api->getAvailabilityDataForView();

        /* Responder con la información de la API, si es que es JSON */
        if ($request->has('apiId')) {
            /* Disponibilidad de una sola habitación */
            return response()->json([
                'roomData' => $roomsData['rooms']->where('api_id', $request->apiId)->first(),
                'betweenDates' => $roomsData['betweenDates'],
            ]);
        }else{
            /* Disponibilidad de todas las habitaciones */
            return response()->json($roomsData);
        }
    }

    /**
     * Método para retornar las fechas que no están disponibles en todo el hotel (sitio actual)
     * @param  Request $request Los datos que el usuario solicitó, incluida la localzación
     * @return Response         Respuesta a mostrar
     */
    public function blackoutDates(Request $request)
    {
        /* Espacio para la información de retorno */
        $retorno;

        /* Establecer sitio para usos futuros */
        $site = Site::current();

        /* Validar la información */
        $request->validate([
            'startDate' => 'nullable|date',
            'endDate' => 'nullable|date|after:startDate',
            'dateFormat' => 'nullable|string'
        ]);

        try {
            $retorno = $this->availableRepository->getTotalBlackoutDates($request->startDate, $request->endDate, $request->dateFormat, $site);
        } catch (Exception $e) {
            return response()->json(null, 500); /* Error en servidor */
        } catch (Throwable $t) {
            return response()->json(null, 422); /* Petición no ejecutable */
        }

        /* Validación exitosa. Continuar */
        return response()->json(
            $retorno
        );
    }

    /**
     * Método para retornar el especial (oferta) activo en el hotel (sitio actual)
     * @param  Request $request Los datos que el usuario solicitó, incluida la localización
     * @return Response         Respuesta a mostrar
     */
    public function special(Request $request)
    {
        /* Establecer sitio para usos futuros */
        $site = Site::current();

        /* Validar la información */
        $request->validate([
            'language' => [
                'required',
                Rule::in(app()->make('translatable.locales')->all())
            ],
            'slug' => [
                'required'
            ],
        ]);

        /* Establecer lenguaje */
        app()->setLocale($request->language);

        try {
            $retorno = $this->pageRepository->getActiveSpecialsBySite($site)->whereTranslation('slug', $request->slug, $request->language)->with(['special', 'mediafiles.type', 'mediafiles.translations'])->first();
        } catch (Exception $e) {
            return response()->json(null, 500); /* Error en servidor */
        } catch (Throwable $t) {
            return response()->json(null, 422); /* Petición no ejecutable */
        }

        /* Validación exitosa. Continuar */
        return response()->json(
            $retorno
        );
    }

    /**
     * Método para retornar los especiales (ofertas) activos en todo el hotel (sitio actual)
     * @param  Request $request Los datos que el usuario solicitó, incluida la localización
     * @return Response         Respuesta a mostrar
     */
    public function specials(Request $request)
    {
        /* Establecer sitio para usos futuros */
        $site = Site::current();

        /* Validar la información */
        $request->validate([
            'language' => [
                'required',
                Rule::in(app()->make('translatable.locales')->all())
            ],
        ]);

        /* Establecer lenguaje */
        app()->setLocale($request->language);

        try {
            $retorno = $this->pageRepository->getActiveSpecialsBySite($site)->translatedIn($request->language)->with(['special', 'mediafiles.type', 'mediafiles.translations'])->defaultOrder()->get();
        } catch (Exception $e) {
            return response()->json(null, 500); /* Error en servidor */
        } catch (Throwable $t) {
            return response()->json(null, 422); /* Petición no ejecutable */
        }

        /* Validación exitosa. Continuar */
        return response()->json(
            $retorno
        );
    }

    /**
     * Método para retornar los restaurantes activos en todo el hotel (sitio actual)
     * @param  Request $request Los datos que el usuario solicitó, incluida la localización
     * @return Response         Respuesta a mostrar
     */
    public function restaurants(Request $request)
    {
        /* Espacio para la información de retorno */
        $retorno = collect();

        /* Establecer sitio para usos futuros */
        $site = Site::current();

        /* Validar la información */
        $request->validate([
            'language' => [
                'required',
                Rule::in(app()->make('translatable.locales')->all())
            ],
        ]);

        /* Establecer lenguaje */
        app()->setLocale($request->language);

        try {
            /** @var Collection|Page[] Lista cruda de páginas con restaurante */
            $rawPages = $this->pageRepository->getRestaurantsBySite($site)->translatedIn($request->language)->with(['restaurant', 'mediafiles.type', 'mediafiles.translations'])->get();

            /* 2020-05-27: El modelo Site revisa la petición para construir la dirección */
            $fullDomain = $site->base_url;

            $retorno = $rawPages->map(function($page) use ($fullDomain, $request) {
                /** @var mixed[] Propiedades de la página */
                $owned = $page->only([
                    'name',
                    'excerpt',
                    'perman_link',
                ]);

                /** @var Mediafile|null Primer medio que concuerde con thumbnail  */
                $ownedThumbnail = $page->mediafiles->filter(function($mediafile) use ($request) {
                    if (!is_null($mediafile->type)) {
                        return ($mediafile->type->type == "thumbnail") && ($mediafile->hasTranslation($request->language));
                    }

                    /* El medio es nulo, por lo que no puede ser thumbnail */
                    return false;
                })->first();

                /** @var string|null URL del thumbnail */
                $owned['thumbnail'] = (!is_null($ownedThumbnail))
                    ? implode([
                          $fullDomain,
                          $ownedThumbnail->path,
                      ])
                    : null;

                /* Añadir el dominio y lenguaje al permalink */
                $owned["perman_link"] = implode([
                    $fullDomain,
                    $owned["perman_link"],
                ]);

                /** @var mixed[] Propiedades del restaurante asociado */
                $referred = $page->restaurant->only([
                    'title',
                    'subtitle',
                    'body',
                    'menus',
                ]);

                /* Crear arreglo vacío, para mantener consistencia */
                if(is_null($referred['menus'])) {
                    $referred['menus'] = [];
                }else if(is_array($referred['menus'])){
                    /* Añadir dirección completa */
                    foreach ($referred['menus'] as &$menu) {
                        $menu['path'] = implode([
                            Str::finish($fullDomain, '/'),
                            $menu['path'],
                        ]);
                    }
                }

                /* Añadir la mezcla */
                return array_merge($owned, $referred);
            });
        } catch (Exception $e) {
            return response()->json(null, 500); /* Error en servidor */
        } catch (Throwable $t) {
            return response()->json(null, 422); /* Petición no ejecutable */
        }

        /* Validación exitosa. Continuar */
        return response()->json(
            $retorno
        );
    }

    /**
     * Método para retornar las habitaciones activas en todo el hotel (sitio actual)
     * @param  Request $request Los datos que el usuario solicitó, incluida la localización
     * @return Response         Respuesta a mostrar
     */
    public function rooms(Request $request)
    {
        /* Espacio para la información de retorno */
        $retorno = collect();

        /* Establecer sitio para usos futuros */
        $site = Site::current();

        /* Validar la información */
        $request->validate([
            'language' => [
                'required',
                Rule::in(app()->make('translatable.locales')->all())
            ],
        ]);

        /* Establecer lenguaje */
        app()->setLocale($request->language);

        try {
            /** @var Collection|Page[] Lista cruda de páginas con habitación */
            $rawPages = $this->pageRepository->getRoomsBySite($site)->translatedIn($request->language)->with(['room', 'parent', 'mediafiles.type', 'mediafiles.translations'])->get();

            /* 2020-05-27: El modelo Site revisa la petición para construir la dirección */
            $fullDomain = $site->base_url;

            $retorno = $rawPages->map(function($page) use ($fullDomain, $request) {
                /** @var mixed[] Propiedades de la página */
                $owned = $page->only([
                    'name',
                    'excerpt',
                    'perman_link',
                ]);

                /* Añadir las propiedades del padre */
                $owned['parent'] = $page->parent->only([
                    'name',
                    'excerpt',
                    'perman_link', /* Para identificar mismo padre sin exponer el ID */
                ]);

                /** @var Mediafile|null Primer medio que concuerde con thumbnail  */
                $ownedThumbnail = $page->mediafiles->filter(function($mediafile) use ($request) {
                    if (!is_null($mediafile->type)) {
                        return ($mediafile->type->type == "thumbnail") && ($mediafile->hasTranslation($request->language));
                    }

                    /* El medio es nulo, por lo que no puede ser thumbnail */
                    return false;
                })->first();

                /** @var string|null URL del thumbnail */
                $owned['thumbnail'] = (!is_null($ownedThumbnail))
                    ? implode([
                          $fullDomain,
                          $ownedThumbnail->path,
                      ])
                    : null;

                /* Añadir el dominio y lenguaje al permalink */
                $owned["perman_link"] = implode([
                    $fullDomain,
                    $owned["perman_link"],
                ]);

                $owned['parent']['perman_link'] = implode([
                    $fullDomain,
                    $owned['parent']['perman_link'],
                ]);

                /** @var mixed[] Propiedades de la habitación asociada */
                $referred = $page->room->only([
                    'description',
                ]);

                /* Añadir la mezcla */
                return array_merge($owned, $referred);
            });
        } catch (Exception $e) {
            return response()->json(null, 500); /* Error en servidor */
        } catch (Throwable $t) {
            return response()->json(null, 422); /* Petición no ejecutable */
        }

        /* Validación exitosa. Continuar */
        return response()->json(
            $retorno
        );
    }

    /**
     * Método para retornar las páginas activas (sitio actual) por permalinks en un lenguaje
     * @param  Request $request Los datos que el usuario solicitó, incluida la localización
     * @return Response         Respuesta a mostrar
     */
    public function pagesByPermalinks(Request $request)
    {
        /* Establecer sitio para usos futuros */
        $site = Site::current();

        /* Validar la información */
        $request->validate([
            'language' => [
                'required',
                Rule::in(app()->make('translatable.locales')->all())
            ],
        ]);

        /* Establecer lenguaje */
        app()->setLocale($request->language);

        try {
            $retorno = $this->pageRepository->bySite($site)->whereTranslation('status', 'published', $request->language)->where(function ($query) use ($request) {
                /* Obtener los valores de los slugs */
                $permalinks = $request->input('permalinks');

                for ($i = 0; $i < count($permalinks); $i++) {
                    /* Construir el URL como viene en la base de datos */
                    $prefixed_permalink = Str::start($permalinks[$i], '/');

                    if ($i == 0) {
                        /* Debe de haber al menos un valor en slugs */
                        $query->whereTranslation('perman_link', $prefixed_permalink, $request->language);
                        continue;
                    }
                    
                    /* Hay más de un elemento: agregarlo como "OR" */
                    $query->orWhereTranslation('perman_link', $prefixed_permalink, $request->language);
                }
            })->with(['mediafiles.type', 'mediafiles.translations'])->get();
        } catch (Exception $e) {
            return response()->json(null, 500); /* Error en servidor */
        } catch (Throwable $t) {
            return response()->json(null, 422); /* Petición no ejecutable */
        }

        /* Validación exitosa. Continuar */
        return response()->json(
            $retorno
        );
    }

    /**
     * Método protegido para cargar relaciones al modelo
     *
     * El modelo original será modificado
     *
     * @param  Page   $page La página en donde deben ser cargados.
     * @return void
     */
    protected function loadRelationsToModel(Page &$page)
    {
        $page
            ->loadMissing([
                'cards' => function ($query) {
                    $query->orderBy('priority', 'asc');
                }
            ])
            ->loadMissing([
                'cards.translations', 'mediafiles.type', 'room.amenities', 'special', 'restaurant', 'site', 'tour',
                'children.mediafiles.type', 'descendants.mediafiles.type', 'ancestors.translations',
                'translations', 'galleries.items.mediafile', 'PageRelations.page.mediafiles.type',
            ]);
    }

    /**
     * Método protegido para el cálculo de reservaciones y retorno de
     * variables necesarias para la vista
     * @param string[]  $requestData  Datos filtrados que vienen de la vista
     * @return mixed[]                Variables necesarias para reservaciones
     */
    protected function getReservationRequestDetails(array $requestData)
    {
        /* INICIA: NUEVA IMPLEMENTACIÓN DE APIs DESACOPLADAS */

        /* Necesario resolver aquí y no en el constructor */
        $api = App::make(ApiAdapter::class);

        /* Establecer datos necesarios */
        $api
            ->setRoomsData($requestData['rooms'])
            ->setLanguage($requestData['language'])
            ->setCurrency($requestData['currency'])
            ->setArrivalDate($requestData['arrivalDate'], $this->defaultDateFormat)
            ->setDepartureDate($requestData['departureDate'], $this->defaultDateFormat);

        /* Obtener habitaciones con información de disponibilidad */
        return $api->getAvailabilityDataForView();

        /* TERMINA: NUEVA IMPLEMENTACIÓN DE APIs DESACOPLADAS */
    }

    /**
     * Método protegido para retornar un validador de solicitud de precios
     * y disponibilidad para habitación
     * @param  Request   $request   La solicitud a revisar
     * @return Validator            El validador
     */
    protected function createPriceRequestValidator(Request $request)
    {
        return Validator::make($request->all(), [
            'rooms' => 'required|array',
            'rooms.*.adults' => 'required|integer|between:1,10',
            'rooms.*.childAges' => 'sometimes|array',
            'rooms.*.childAges.*' => 'integer|between:0,12',
            'arrivalDate' => [
                'required',
                'date',
                'after:' . date($this->defaultDateFormat, time())
            ],
            'departureDate' => 'required|date|after:arrivalDate',
            'language' => [
                'required',
                Rule::in(app()->make('translatable.locales')->all())
            ],
            'currency' => [
                'required',
                Rule::in(['MXN', 'USD'])
            ],
            'apiId' => [
                'sometimes',
                'required',
                'exists:room_api_codes,api_code'
            ]
        ]);
    }

    /**
     * Método protegido para retornar datos para una búsqueda predeterminada
     * @param string        $language   Identificador de localización solicitada
     *                                  (Ej. 'en', 'es'...)
     * @param string|null   $dateFormat Formato en el que las fechas estarán
     * @return mixed[]              Arreglo de datos para búsqueda
     */
    protected function createDefaultSearchValues($language, $dateFormat = null)
    {
        /* Usar el formato predeterminado de este controlador */
        if (is_null($dateFormat)) {
            $dateFormat = $this->defaultDateFormat;  /* mm/dd/yyyy */
        }

        return [
            'rooms' => [
                [
                    'adults' => 2,
                ]
            ],
            'arrivalDate' => date($dateFormat, time()),
            'departureDate' => date($dateFormat, strtotime('+2 day')),
            'language' => $language,
            'currency' => ($language == 'es' ? 'MXN' : 'USD'),
        ];
    }
}
