<?php

namespace AvengersMG\MGCms2019\App\Http\Controllers\Backend\Users;

use AvengersMG\MGCms2019\App\Cms\Roles\Repositories\RoleInterface;
use AvengersMG\MGCms2019\App\Cms\Sites\Repositories\SiteInterface;
use AvengersMG\MGCms2019\App\Cms\Users\Repositories\UserInterface;
use AvengersMG\MGCms2019\App\Cms\Users\Requests\StoreUserRequest;
use AvengersMG\MGCms2019\App\Cms\Users\Requests\UpdateUserRequest;
use AvengersMG\MGCms2019\App\Cms\Sites\Site;
use AvengersMG\MGCms2019\App\Cms\Users\User;
use AvengersMG\MGCms2019\App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class UsersController extends Controller
{
    /** @var RoleInterface Repositorio de objetos Role */
    private $userRepository;

    /** @var UserInterface Repositorio de objetos User */
    private $roleRepository;

    /** @var SiteInterface Repositorio de objetos Site */
    private $siteRepository;

    /**
     * Constructor de controlador
     *
     * @param RoleInterface     $role    Repositorio de modelo Role
     * @param UserInterface     $user    Repositorio de modelo User
     * @param SiteInterface     $site    Repositorio de modelo Site
     * @return void
     */
    public function __construct(
        RoleInterface $role,
        UserInterface $user,
        SiteInterface $site
    ) {
        /* Provisionar controlador */
        $this->roleRepository = $role;
        $this->userRepository = $user;
        $this->siteRepository = $site;
    }

    /**
     * Método público para mostrar la vista de lista de usuarios
     *
     * @return Response  Dirige a vista de lista
     */
    public function index()
    {
        /* Buscar la autorización */
        $this->authorize('viewAny', User::class);

        $users = $this->userRepository->list();
        return view('MGCms2019::admin.users.index', compact('users'));
    }

    /**
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function create()
    {
        /* Buscar la autorización */
        $this->authorize('create', User::class);

        $roles = $this->roleRepository->list();

        $sites = $this->siteRepository->list();

        return view('MGCms2019::admin.users.create', compact('roles', 'sites'));
    }

    /**
     * @param StoreUserRequest $request app/Cms/Users/Requests/StoreUserRequest
     * @return mixed $user is a list of users
     */
    public function store(StoreUserRequest $request)
    {
        $return_route = 'users.index';

        /* Buscar la autorización */
        $this->authorize('create', User::class);

        $user = $this->userRepository->create($request->all());
        $user->syncRoles([$request->role]);
        $user->sites()->sync($request->sites);

        $requesting_user = Auth::user();

        /* Si el usuario no puede listar usuarios, redirigir a la página de creación */
        if ($requesting_user->cant('viewAny', User::class)) {
            $return_route = 'users.create';
        }

        /**
         * este metodo esta en app/Http/Controllers/Controller
         * recibe como primer parametro un objeto, segundo el nombre de la ruta
         * y tercero el mensaje
         *
         * 2019-06-20: el segundo parámetro también puede ser un arreglo con
         * parámetros para la ruta
         */
        return $this->responseController($user, $return_route, "El usuario {$request->username} se creó correctamente");
    }

    public function show(User $user)
    {
        /* Buscar la autorización */
        $this->authorize('view', $user);

        //
    }

    public function edit(User $user)
    {
        /* Buscar la autorización */
        $this->authorize('update', $user);

        $roles = $this->roleRepository->list();

        $sites = $this->siteRepository->list();

        $user_sites = $user->sites->pluck('id')->toArray();

        return view('MGCms2019::admin.users.edit', compact('user', 'roles', 'user_sites', 'sites'));
    }

    /**
     * @param UpdateUserRequest $request app/Cms/Users/Requests/UpdateUserRequest.php
     * @param User $user instance of model User
     * @return mixed
     */

    public function update(UpdateUserRequest $request, User $user)
    {
        $return_route = 'users.index';

        /* Buscar la autorización */
        $this->authorize('update', $user);

        $success = $this->userRepository->update($request->all(), $user);

        $requesting_user = Auth::user();

        /* Si tiene permiso de cambiar su propio rol, permitir */
        if($request->has('role') && $requesting_user->can('updateOwnRole', $user)){
            $user->syncRoles([$request->role]);
        }

        /* Si tiene permiso de cambiar asignación de sitios, permitir */
        if($request->has('sites') && $requesting_user->can('updateOwnSites', User::class)){
            $user->sites()->sync($request->sites);
        }

        /* Si el usuario no puede listar usuarios, redirigir a la página de edición */
        if ($requesting_user->cant('viewAny', User::class)) {
            $return_route = ['users.edit', $user];
        }

        /**
         * este metodo esta en app/Http/Controllers/Controller
         * recibe como primer parametro un objeto, segundo el nombre de la ruta
         * y tercero el mensaje
         *
         * 2019-06-20: el segundo parámetro también puede ser un arreglo con
         * parámetros para la ruta
         */
        return $this->responseController($success, $return_route, "El usuario {$user->name} fue modificado");
    }

    /**
     * @param User $user
     * @return mixed
     */
    public function destroy(User $user)
    {
        /* Buscar la autorización */
        $this->authorize('delete', $user);

        /* Nombre de usuario, para uso futuro */
        $user_name = $user->name;

        // fuerza la destruccion del usuario para evitar el softdelete
        $deleted = $user->forceDelete();

        if ($deleted) {
            /* False en el método hará un redirect()->back() */
            return $this->responseController(false, null, "El usuario {$user_name} fue eliminado");
        }

        /* Usuario no eliminado. Regresar con error */
        return $this->responseController(false, null);
    }
}
