<?php

namespace AvengersMG\MGCms2019\App\Http\Controllers\Backend\Tours;

use AvengersMG\MGCms2019\App\Cms\Components\Tours\Tour;
use AvengersMG\MGCms2019\App\Cms\Components\Tours\ToursInterface;
use AvengersMG\MGCms2019\App\Cms\Pages\Page;
use AvengersMG\MGCms2019\App\Cms\Sites\Repositories\SiteInterface;
use AvengersMG\MGCms2019\App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class ToursController extends Controller
{
    protected $model;
    protected $site;
    protected $view = 'MGCms2019::admin.components.tours.';

    public function __construct(
          ToursInterface $tours,
          SiteInterface $sites
    )
    {
        $this->model = $tours;
        $this->site = $sites;
    }

    public function create(Page $page, $lang)
    {
        /* Buscar la autorización */
        $this->authorize('update', $page);

        return view($this->view . 'create', compact('page', 'lang'));
    }

    public function store(Request $request)
    {
        /* Página asociada */
        $page = Page::find($request->page_id);

        /* Buscar la autorización */
        $this->authorize('update', $page);

        $this->model->create($request->all());

        return redirect()->route('pages.show', [$page->id, $request->locale])->withSuccess('Creación correcta del tour');
    }

    public function update(Request $request, Tour $tour)
    {
        $messageSuccess = 'Actualización de tour correcta.';
        $messageFailure = 'Ups. Algo salió mal con la actualización de tour.';

        /* Página asociada */
        $page = $tour->page;

        /* Buscar la autorización */
        $this->authorize('update', $page);

        /**
         * Bandera de indicación si el usuario puede acceder al listado
         * @var boolean
         */
        $canViewAny = Auth::user()->can('viewAny', [Page::class, $page->site]);

        /**
         * Bandera de indicación si la página pertenece al sitio actual
         * @var boolean
         */
        $isCurrentSite = ($page->site->id == $this->site->current()->id);

        /**
         * Información de ruta para los casos en que el usuario tenga o no
         * permiso para acceder al listado
         * @var string|mixed[]
         */
        $routeRetorno = ($canViewAny)
            ? ($isCurrentSite)
                ? 'pages.index-from-current'
                : ['pages.index', [$page->site]] /* Establecer respuesta de retorno. Si página pertenece a sitio actual, esconder URL */
            : ['pages.show', [$page->id, $request->locale]]; /* page.show está usado para editar, por lo que comparten el mismo permiso y no es necesario revisar */

        /*
            ADVERTENCIA: El método prepareUpdate() en el repositorio
            no genera retorno, por lo que no es posible avisar de fallas
         */

        $this->model->prepareUpdate($request, $tour);

        if ($request->wantsJson()) {
            /* Retornar objeto actualizado */
            return $tour->fresh();
        }

        /* La petición no solicita JSON: Retornar al listado de páginas  */
        return $this->responseController(true, $routeRetorno, $messageSuccess);
    }

    /**
     * Método público para eliminar un tour y deshacer asociaciones
     *
     * @param  Tour     $tour Tour a eliminar
     * @param  string   $lang Idioma inyectado desde la página
     * @return Response       Redirección a página asociada
     */
    public function destroy(Tour $tour, $lang)
    {
        /* Página asociada */
        $page = $tour->page;

        /* Buscar la autorización */
        $this->authorize('update', $page);

        /* Eliminar modelo y traducciones */
        $this->model->deleteAll($tour);

        /* Redirigir a la página que tuvo asignada */
        return redirect()->route('pages.show', [$page->id, $lang]);
    }
}
