<?php
namespace AvengersMG\MGCms2019\App\Http\Controllers\Backend\Specials;

use AvengersMG\MGCms2019\App\Cms\Components\Specials\Special;
use AvengersMG\MGCms2019\App\Cms\Components\Specials\SpecialInterface;
use AvengersMG\MGCms2019\App\Cms\Pages\Page;
use AvengersMG\MGCms2019\App\Cms\Sites\Repositories\SiteInterface;
use AvengersMG\MGCms2019\App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class SpecialController extends Controller
{
    /** @var view ruta de la carpeta donde se encuentran las vistas  */
    protected $view = 'admin.components.specials.';

    /** @var SpecialsRepository objeto del repositorio de Special */
    protected $specialsRepository;

    /** @var SiteRepository objeto del repositorio de Special */
    protected $siteRepository;

    /**
     * Constructor del controlador
     *
     * @param SpecialInterface $specialInterface Repositorio de modelo Special
     * @param SiteInterface    $siteInterface    Repositorio de modelo Site
     */
    public function __construct(
        SpecialInterface $specialInterface,
        SiteInterface $siteInterface
    )
    {
        /* Provisionar controlador con los repositorios */
        $this->specialsRepository = $specialInterface;
        $this->siteRepository = $siteInterface;
    }

    /**
     * @param $page Vinculacion impliicita del model Page
     * @param $lang Variable de lenguage
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function create(Page $page, $lang)
    {
        /* Buscar la autorización */
        $this->authorize('update', $page);

        /** setear lenguaje */
        app()->setLocale($lang);
        return view('MGCms2019::'.$this->view.'create', compact('page', 'lang'));
    }

    /**
     * Método público de guardado de datos para nueva página
     * @param $request
     * @return redirecciona a la vista al metodo show
     */
    public function store(Request $request)
    {
        //dd($request->all());
        /* Página asociada */
        $page = Page::find($request->page_id);

        /* Buscar la autorización */
        $this->authorize('update', $page);

        $this->specialsRepository->store($request);

        return redirect()->route('pages.show', [$page->id, $request->locale]);
    }

    /**
     * Método público de actualización de un special
     * @param  \Illuminate\Http\Request  $request
     * @param $special Vinculacion impliicita del model Special
     */
    public function update(Request $request, Special $special)
    {
        //return $request->all();
        $messageSuccess = 'Actualización de especial correcta.';
        $messageFailure = 'Ups. Algo salió mal con la actualización de especial.';

        /* Página asociada, para usos futuros */
        $page = $special->page;

        /* Buscar la autorización */
        $this->authorize('update', $page);

        /**
         * Bandera de indicación si el usuario puede acceder al listado
         * @var boolean
         */
        $canViewAny = Auth::user()->can('viewAny', [Page::class, $page->site]);

        /**
         * Bandera de indicación si la página pertenece al sitio actual
         * @var boolean
         */
        $isCurrentSite = ($page->site->id == $this->siteRepository->current()->id);

        /**
         * Información de ruta para los casos en que el usuario tenga o no
         * permiso para acceder al listado
         * @var string|mixed[]
         */
        $routeRetorno = ($canViewAny)
            ? ($isCurrentSite)
                ? 'pages.index-from-current'
                : ['pages.index', [$page->site]] /* Establecer respuesta de retorno. Si página pertenece a sitio actual, esconder URL */
            : ['pages.show', [$page->id, $request->locale]]; /* page.show está usado para editar, por lo que comparten el mismo permiso y no es necesario revisar */

        /** @var Special Objeto actualizado  */
        $success = $this->specialsRepository->updateSpecial($request, $special);

        if ($success) {
            if ($request->wantsJson()) {
                return $success;
            }

            return $this->responseController(true, $routeRetorno, $messageSuccess);
        }else if ($request->wantsJson()) {
            return response()->json($messageFailure, 500);
        }else if($canViewAny){
            /* Establecer respuesta de retorno. Si página pertenece a sitio actual, esconder URL */
            return $this->responseController(true, $routeRetorno, $messageFailure);
        }

        /* No resultó. Retornar redirección back() con mensaje */
        return $this->responseController(false, null, $messageFailure);
    }

    /**
     * Método público para eliminar un especial y deshacer asociaciones
     *
     * @param  Special     $special Especial a eliminar
     * @param  string      $lang    Idioma inyectado desde la página
     * @return Response             Redirección a página asociada
     */
    public function destroy(Special $special, $lang)
    {
        /* Página asociada, para uso futuro */
        $page = $special->page;

        /* Buscar la autorización */
        $this->authorize('update', $page);

        /* Eliminar modelo y traducciones */
        $this->specialsRepository->deleteAll($special);

        /* Redirigir a la página que tuvo asignada */
        return redirect()->route('pages.show', [$page->id, $lang]);
    }
}
