<?php

namespace AvengersMG\MGCms2019\App\Http\Controllers\Backend\Roles;

use AvengersMG\MGCms2019\App\Cms\Roles\Repositories\RoleInterface;
use AvengersMG\MGCms2019\App\Http\Controllers\Controller;
use Spatie\Permission\Models\Permission;
use Spatie\Permission\Models\Role;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class RolesController extends Controller
{

    /**
     * IMPORTANT
     * Se utiliza la libreria de spatie para los permisos y roles.
     * https://github.com/spatie/laravel-permission
     */

    /** @var roleRepository Repositorio de objetos Page */
    protected $roleRepository;

    /**
     * @param roleInterface      $roleInterface     Repositorio de modelo Role
     */
    public function __construct(RoleInterface $roleInterface)
    {
        $this->roleRepository = $roleInterface;
    }

    /**
     * Metodo publico que retorna la vista donde estan listados los roles.
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function index()
    {
        /* Buscar la autorización */
        $this->authorize('viewAny', Role::class);

        $roles = $this->roleRepository->list();
        return view('MGCms2019::admin.users.roles.index', compact('roles'));
    }

    /**
     * Metodo publico para la creacion de un rol
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function create()
    {
        /* Buscar la autorización */
        $this->authorize('create', Role::class);

        return view('MGCms2019::admin.users.roles.create');
    }

    /**
     * @param  \Illuminate\Http\Request  $request
     * @return instance of Illuminate\Routing\Redirector
     */
    public function store(Request $request)
    {
        $return_route = 'roles.index';

        /* Buscar la autorización */
        $this->authorize('create', Role::class);

        $request->validate(['name' => 'required|unique:roles,name']);

        $success = $this->roleRepository->create($request->all());

        /* Si el usuario no tiene permiso de listar roles, retornar a creación */
        if (Auth::user()->cant('viewAny', Role::class)) {
            $return_route = 'roles.create';
        }

        return $this->responseController($success, $return_route, "El rol {$request->name} se creó correctamente");
    }

     /**
     * @param  role Vinculacion implicita del modelo Role
     * @return instance of Illuminate\Routing\Redirector
     */
    public function show(Role $role)
    {
        /*
          Buscar la autorización

          ADVERTENCIA: Este método está siendo utilizado para editar permisos en lugar de mostrar detalles de rol, así que es necesario buscar autorización de edición
         */
        $this->authorize('update', $role);

        $permissions = Permission::all();
        $role_permissions = $role->permissions->pluck('name')->toArray();
        return view('MGCms2019::admin.users.roles.show', compact( 'role','permissions', 'role_permissions'));
    }

    /**
     * @param  role Vinculacion implicita del modelo Role
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function edit(Role $role)
    {
        /* Buscar la autorización */
        $this->authorize('update', $role);

        return view('MGCms2019::admin.users.roles.edit', compact('role'));
    }

    /**
     * @param  \Illuminate\Http\Request  $request
     * @param  role Vinculacion implicita del modelo Role
     * @return instance of Illuminate\Routing\Redirector
     */
    public function update(Request $request, Role $role)
    {
        $return_route = 'roles.index';

        /* Buscar la autorización */
        $this->authorize('update', $role);

        $success = $this->roleRepository->update($request, $role);

        /* Si el usuario no tiene permiso de listar roles, retornar a edición */
        if (Auth::user()->cant('viewAny', Role::class)) {
            $return_route = ['roles.edit', $role->id];
        }

        return $this->responseController($success, $return_route, "El rol {$role->name} fue modificado");
    }

    /**
     * Metodo publico para eliminar un role
     * @param  role Vinculacion implicita del modelo Role
     * @return instance of Illuminate\Routing\Redirector
     */
    public function destroy(Role $role)
    {
        /* Buscar la autorización */
        $this->authorize('delete', $role);

        $role_name = $role->name;

        $deleted = $this->roleRepository->delete($role);

        if ($deleted) {
            /* false hace redirect()->back()  */
            return $this->responseController(false, null, "El rol {$role_name} fue eliminado");
        } else {
            return $this->responseController(false, null);
        }
    }

    /**
     * Metodo publico de asignacion de permisos a un rol
     * @param  \Illuminate\Http\Request  $request
     * @param  role Vinculacion implicita del modelo Role
     * @return instance of Illuminate\Routing\Redirector
     */
    public function attachPermissions(Request $request, Role $role)
    {
        $return_route = 'roles.index';

        /* Buscar la autorización */
        $this->authorize('update', $role);

        /** Sincroniza la relacion entre los roles y los permisos */
        $role->syncPermissions($request->permissions);

        /* Si el usuario no tiene permiso de listar roles, retornar a asignación de permisos */
        if (Auth::user()->cant('viewAny', Role::class)) {
            $return_route = ['roles.show', $role->id];
        }

        return $this->responseController(true, $return_route, "Se asignaron los permisos correctamente. ");
    }

}
