<?php

namespace AvengersMG\MGCms2019\App\Http\Controllers\Backend\Restaurants;

use \App;
use AvengersMG\MGCms2019\App\Cms\Components\Restaurants\Requests\RestaurantRequest;
use AvengersMG\MGCms2019\App\Cms\Components\Restaurants\Restaurant;
use AvengersMG\MGCms2019\App\Cms\Components\Restaurants\RestaurantsInterface;
use AvengersMG\MGCms2019\App\Cms\Components\Restaurants\Service;
use AvengersMG\MGCms2019\App\Cms\Pages\Page;
use AvengersMG\MGCms2019\App\Cms\Sites\Repositories\SiteInterface;
use AvengersMG\MGCms2019\App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Auth;

class RestaurantController extends Controller
{
    protected $view = 'MGCms2019::admin.components.restaurants.';
    protected $model;
    protected $site;
    /**
     * [__construct Constructor de clase]
     * @param RestaurantsInterface $restaurant inyección de interface para implementación métodos
     */
    function __construct(
        RestaurantsInterface $restaurant,
        SiteInterface $site
    )
    {
        $this->model = $restaurant;
        $this->site = $site;
    }
    /**
     * [create]
     * @param  Page   $page Contiene el modelo de p{agina}
     * @param  String $lang contiene el idioma en:es
     * @return Response  Retorna la vista para crear un restaurant con $page, $lang, $services
     */
    public function create(Page $page, $lang)
    {
        /* Buscar la autorización */
        $this->authorize('update', $page);

        /* Cambia el idioma de la aplicación por el que recibe del parámetro $lang */
        app()->setLocale($lang);
        /* obtine todos los servicios del modelo Service */
        $services = Service::all();
        /* retorna a la vista de creación de restaurants con la data necesaria */
        return view($this->view.'create', compact('page', 'lang', 'services'));
    }
    /**
     * [store recibe los datos desde la vista miente un formulario]
     * @param  RestaurantRequest $request Recibe los datos desde la vista y lo retorna ya validados
     * @return Response                     [description]
     */
    public function store(RestaurantRequest $request)
    {
        /* Página asociada */
        $page = Page::find($request->page_id);

        /* Buscar la autorización */
        $this->authorize('update', $page);

        /* almacena todos los datos obtenidos de request*/
        $data = $request->all();
        //dd($data);
        if (isset($request->menus)) {
            $menu_array = [];
            foreach ($data['menus'] as $value) {
                $name = $value->getClientOriginalName();
                $path = $value->storeAs('mediafiles/restaurants/'.$request->page_id, $name, 'MGCms2019');
                $name = explode('.', $value->getClientOriginalName());
                $name = str_slug($name[0]);
                $menu = ["name"=> $name, "path" => $path];
                array_push($menu_array, $menu);
            }
            $data['menus'] = json_encode($menu_array);
        }

        $this->model->create($data);

        return redirect()->route('pages.show', [$page->id, $request->locale])->withInput()->withSuccess('Creación correcta del restaurante');
    }

    public function show(Restaurant $restaurant, $index, $lang)
    {
        /* Buscar la autorización */
        $this->authorize('view', $restaurant->page);

        App::setlocale($lang);
        $menus = $restaurant->menus;
        unset($menus[$index]);
        $restaurant->menus = json_encode($menus);
        $restaurant->save();
        return $restaurant;
    }

    public function update(Request $request, Restaurant $restaurant)
    {
        $messageSuccess = 'Actualización de restaurante correcta.';
        $messageFailure = 'Ups. Algo salió mal con la actualización de restaurante.';

        /* Página asociada */
        $page = $restaurant->page;

        /* Buscar la autorización */
        $this->authorize('update', $page);

        /**
         * Bandera de indicación si el usuario puede acceder al listado
         * @var boolean
         */
        $canViewAny = Auth::user()->can('viewAny', [Page::class, $page->site]);

        /**
         * Bandera de indicación si la página pertenece al sitio actual
         * @var boolean
         */
        $isCurrentSite = ($page->site->id == $this->site->current()->id);

        /**
         * Información de ruta para los casos en que el usuario tenga o no
         * permiso para acceder al listado
         * @var string|mixed[]
         */
        $routeRetorno = ($canViewAny)
            ? ($isCurrentSite)
                ? 'pages.index-from-current'
                : ['pages.index', [$page->site]] /* Establecer respuesta de retorno. Si página pertenece a sitio actual, esconder URL */
            : ['pages.show', [$page->id, app()->getLocale()]]; /* page.show está usado para editar, por lo que comparten el mismo permiso y no es necesario revisar */

        $data = $request->all();
        if ($request->has('menus')) {
            $menu_array = [];
            foreach ($request->menus as $value) {
                if (is_array($value)) {
                    $name = $value[0];
                    $path = $value[1];
                    $menu = ["name"=> $name, "path" => $path];
                    array_push($menu_array, $menu);
                } else {
                    $name = $value->getClientOriginalName();
                    $path = $value->storeAs('mediafiles/restaurants/'.$request->page_id, $name, 'MGCms2019');
                    $name = explode('.', $value->getClientOriginalName());
                    $name = str_slug($name[0]);
                    $menu = ["name"=> $name, "path" => $path];
                    array_push($menu_array, $menu);
                }
            }

            $data['menus'] = json_encode($menu_array);
        }

        /*
            ADVERTENCIA: El método prepareUpdate() en el repositorio
            no genera retorno, por lo que no es posible avisar de fallas
         */

        $this->model->prepareUpdate($data, $restaurant);

        if ($request->wantsJson()) {
            /* Retornar objeto actualizado */
            return $restaurant->fresh();
        }

        /* La petición no solicita JSON: Retornar al listado de páginas  */
        return $this->responseController(true, $routeRetorno, $messageSuccess);
    }

    /**
     * Método público para eliminar un restaurante y deshacer asociaciones
     *
     * @param  Restaurant  $restaurant Restaurante a eliminar
     * @param  string     $lang        Idioma inyectado desde la página
     * @return Response                Redirección a página asociada
     */
    public function destroy(Restaurant $restaurant, $lang)
    {
        /* Página asociada */
        $page = $restaurant->page;

        /* Buscar la autorización */
        $this->authorize('update', $page);

        /* Eliminar modelo y traducciones */
        $this->model->deleteAll($restaurant);

        /* Redirigir a la página que tuvo asignada */
        return redirect()->route('pages.show', [$page->id, $lang]);
    }
}
