<?php

namespace AvengersMG\MGCms2019\App\Http\Controllers\Backend\Permisions;

use AvengersMG\MGCms2019\App\Http\Controllers\Controller;
use AvengersMG\MGCms2019\App\Cms\Permissions\Repositories\PermisionInterface;
use AvengersMG\MGCms2019\App\Cms\Permissions\Repositories\PermissionInterface;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Spatie\Permission\Models\Permission;

class PermisionsController extends Controller
{

    /**
     * IMPORTANT
     * Se utiliza la libreria de spatie para los permisos y roles.
     * https://github.com/spatie/laravel-permission
     */

    /** @var PermissionInterface Repositorio de objetos Permission */
    protected $permissionRepository ;

    /**
     * @param PermissionInterface      $permissionInterface     Repositorio de modelo Permission
     */
    public function __construct(PermissionInterface $permissionInterface)
    {
        /* Provisionar controlador con los repositorios */
        $this->permissionRepository = $permissionInterface;
    }

    /**
     * Metodo publico que retorna la vista donde estan listados los permisos.
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function index()
    {
        /* Buscar la autorización */
        $this->authorize('viewAny', Permission::class);

        $permissions = $this->permissionRepository->list();
        return view('MGCms2019::admin.users.permissions.index', compact('permissions'));
    }

     /**
     * Metodo publico para la creacion de un permiso
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function create()
    {
        /* Buscar la autorización */
        $this->authorize('create', Permission::class);

        return view('MGCms2019::admin.users.permissions.create');
    }

    /**
     * @param  \Illuminate\Http\Request  $request
     * @return instance of Illuminate\Routing\Redirector
     */
    public function store(Request $request)
    {
        $return_route = 'permissions.index';

        /* Buscar la autorización */
        $this->authorize('create', Permission::class);

        $request->validate(['name' => 'required|unique:permissions,name']);

        $success = $this->permissionRepository->create($request->all());

        /* Si el usuario no tiene permiso de listar permisos, retornar a creación */
        if (Auth::user()->cant('viewAny', Permission::class)) {
            $return_route = 'permissions.create';
        }

        return $this->responseController($success, $return_route, "El permiso {$request->name} se creó correctamente");
    }

    /**
     * @param  permission Vinculacion implicita del modelo Permission
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function edit(Permission $permission)
    {
        /* Buscar la autorización */
        $this->authorize('update', $permission);

        return view('MGCms2019::admin.users.permissions.edit', compact('permission'));
    }

    /**
     * @param  \Illuminate\Http\Request  $request
     * @param  permission Vinculacion implicita del modelo Permission
     * @return instance of Illuminate\Routing\Redirector
     */
    public function update(Request $request, Permission $permission)
    {
        $return_route = 'permissions.index';

        /* Buscar la autorización */
        $this->authorize('update', $permission);

        $success = $this->permissionRepository->update($request, $permission);

        /* Si el usuario no tiene permiso de listar permisos, retornar a edición */
        if (Auth::user()->cant('viewAny', Permission::class)) {
            $return_route = ['permissions.edit', $permission->id];
        }

        return $this->responseController($success, $return_route,"El permiso {$permission->name} fue modificado");
    }

    /**
     * Metodo publico para eliminar un role
     * @param  permission Vinculacion implicita del modelo Permission
     * @return instance of Illuminate\Routing\Redirector
     */
    public function destroy(Permission $permission)
    {
        /* Buscar la autorización */
        $this->authorize('delete', $permission);

        /* Obtener el nombre */
        $permission_name = $permission->name;

        $deleted = $this->permissionRepository->delete($permission);

        if ($deleted) {
            return $this->responseController(false, null, "El permiso {$permission_name} fue eliminado");
        } else {
            return $this->responseController(false, null);
        }
    }

}
