<?php

namespace AvengersMG\MGCms2019\App\Http\Controllers\Backend\Pages;

use \Exception;
use \Throwable;
use AvengersMG\MGCms2019\App\Cms\Components\Cards\Repositories\CardInterface;
use AvengersMG\MGCms2019\App\Cms\Components\Galleries\Repositories\GalleryInterface;
use AvengersMG\MGCms2019\App\Cms\Components\Restaurants\Services\ServiceInterface;
use AvengersMG\MGCms2019\App\Cms\Forms\Form;
use AvengersMG\MGCms2019\App\Cms\Forms\FormsInterface;
use AvengersMG\MGCms2019\App\Cms\Mediafiles\Mediafile;
use AvengersMG\MGCms2019\App\Cms\Mediafiles\Repositories\FiletypesInterface;
use AvengersMG\MGCms2019\App\Cms\Mediafiles\Repositories\MediafilesInterface;
use AvengersMG\MGCms2019\App\Cms\Pages\Page;
use AvengersMG\MGCms2019\App\Cms\Pages\Request\PageRequest;
use AvengersMG\MGCms2019\App\Cms\Pages\Request\UpdatePageRequest;
use AvengersMG\MGCms2019\App\Cms\Pages\Request\UpdateSeoPageRequest;
use AvengersMG\MGCms2019\App\Cms\Pages\Request\UpdateTreePageRequest;
use AvengersMG\MGCms2019\App\Cms\Pages\Request\UploadFilePageRequest;
use AvengersMG\MGCms2019\App\Cms\Pages\Repositories\PageInterface;
use AvengersMG\MGCms2019\App\Cms\Sites\Site;
use AvengersMG\MGCms2019\App\Cms\Sites\Repositories\SiteInterface;
use AvengersMG\MGCms2019\App\Http\Controllers\Controller;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;

class PagesController extends Controller
{
    /** @var PageInterface Repositorio de objetos Page */
    private $pageRepository;

    /** @var CardInterface Repositorio de objetos Card */
    private $cardRepository;

    /** @var GalleryInterface Repositorio de objetos Gallery */
    private $galleryRepository;

    /** @var FormInterface Repositorio de objetos Form */
    private $formRepository;

    /** @var ServiceInterface Repositorio de objetos Service */
    private $serviceRepository;

    /** @var FiletypesInterface Repositorio de objetos Service */
    private $filetypeRepository;

    /** @var SiteInterface Repositorio de objetos Site */
    private $siteRepository;

    /** @var MediafilesInterface Repositorio de objetos Mediafile */
    private $mediafileRepository;

    /**
     * Constructor de controlador
     *
     * @param PageInterface       $pageInterface      Repositorio de modelo Page
     * @param CardInterface       $cardInterface      Repositorio de modelo Card
     * @param GalleryInterface    $galleryInterface   Repositorio de modelo Gallery
     * @param FormsInterface      $formInterface      Repositorio de modelo Form
     * @param ServiceInterface    $serviceInterface   Repositorio de modelo Service
     * @param FiletypesInterface  $filetypeInterface  Repositorio de modelo FileType
     * @param SiteInterface       $siteInterface      Repositorio de modelo Site
     * @param MediafilesInterface $mediafileInterface Repositorio de modelo Mediafile
     * @return void
     */
    public function __construct(
        PageInterface $pageInterface,
        CardInterface $cardInterface,
        GalleryInterface $galleryInterface,
        FormsInterface $formInterface,
        ServiceInterface $serviceInterface,
        FiletypesInterface $filetypeInterface,
        SiteInterface $siteInterface,
        MediafilesInterface $mediafileInterface
    ) {
        /* Provisionar controlador con los repositorios */
        $this->pageRepository = $pageInterface;
        $this->cardRepository = $cardInterface;
        $this->galleryRepository = $galleryInterface;
        $this->formRepository = $formInterface;
        $this->serviceRepository = $serviceInterface;
        $this->filetypeRepository = $filetypeInterface;
        $this->siteRepository = $siteInterface;
        $this->mediafileRepository = $mediafileInterface;
    }

    /**
     * Método público para mostrar la vista de lista de páginas de acuerdo al sitio
     *
     * @param  Site     $site Objeto Site inyectado desde la vista
     * @return Response       Dirige a vista de lista
     */
    public function index(Site $site)
    {
        /* Establecer sitio a tratar */
        if (!$site->exists) {
            $site = $this->siteRepository->current();
        }

        if (is_null($site)) {
            /* No existen sitios definidos */
            return $this->responseNoSitesFound();
        }

        /* Buscar la autorización */
        $this->authorize('viewAny', [Page::class, $site]);

        /* Intentar mostrar jerarquía */
        try {
            $pages = $this->pageRepository->toTree($site);
            return view('MGCms2019::admin.pages.index', compact('site', 'pages'));
        } catch (Exception $e) {
            return abort(500, 'Error general');
        }
    }

    /**
     * Método público para mostrar la vista de nueva página
     *
     * @param  Site       $site   Objeto Site inyectado desde la vista
     * @param  Page       $page   Objeto Page padre
     * @return Response           Dirige a vista de creación
     */
    public function create(Site $site, Page $page)
    {
        /** @var Collection|string[] Lista de idiomas con etiqueta */
        $available_languages = collect();

        /** @var boolean Resultado de verificar si la página existe */
        $isPageDefined = $page->exists;

        /* Predeterminar sitio si es que no viene especificado en los parámetros */
        if ($isPageDefined) {
            $site = $page->site;
        } elseif (!$site->exists) {
            /* El ID del sitio es predeterminada del primer sitio */
            $site = $this->siteRepository->current();
        }

        if (is_null($site)) {
            /* No existen sitios definidos */
            return $this->responseNoSitesFound();
        }

        /* Buscar la autorización */
        $this->authorize('create', [Page::class, $site]);

        /* Definir la fuente de lenguajes */
        $languageSource = ($isPageDefined) ? $page : $site;

        /* Recuperar lenguajes */
        $available_languages = $languageSource->translations->pluck('locale', 'locale');

        /* Recuperar variables comunes */
        $returnData = $this->getPageCommonData(new Page(['site_id' => $site->id]));

        /* Incluir variables específicas */
        $returnData = $returnData->merge([
            'parent' => ($isPageDefined) ? $page : null,
            'available_languages' => $available_languages
        ]);

        /* Retornar vista (segundo parámetro DEBE ser arreglo) */
        return view('MGCms2019::admin.pages.create', $returnData->all());
    }

    /**
     * Método público de guardado de datos para nueva página
     *
     * NOTA: Validación en PageRequest@rules
     *
     * @param  Site        $site    Sitio inyectado desde vista
     * @param  PageRequest $request Información de página
     * @return Response             Dirige a vista de lista
     */
    public function store(Site $site, PageRequest $request)
    {
        $current_site = $this->siteRepository->current();

        /* Si no hay sitio inyectado, asignar el actual */
        if (!$site->exists) {
            $site = $current_site;
        }

        if (is_null($site)) {
            /* No existen sitios definidos */
            return $this->responseNoSitesFound();
        }

        /* Buscar la autorización */
        $this->authorize('create', [Page::class, $site]);

        /* Autorizado: Añadir el valor del sitio web a los valores de la solicitud */
        $data = array_merge(
            $request->all(),
            [
                'site_id' => $site->id
            ]
        );

        /* Crear página  */
        $this->pageRepository->create($data);

        /* Redirigir a listardo si permite */
        if (Auth::user()->can('viewAny', [Page::class, $site])) {
            /* Verificar que el sitio fue especificado */
            if ($site->id == $current_site->id) {
                /* No fue especificado. Usar actual */
                return redirect()->route('pages.index-from-current');
            }

            /* Sitio estaba establecido originalmente. Redirigir de acuerdo con sitio enviado */
            return redirect()->route('pages.index', $site);
        }

        /* Usuario no tiene permitido listar páginas. Redirigir a creación */
        return redirect()->route('pages.create', [$site, $page->parent]);
    }

    /**
     * Método público de muestreo (y edición) de Page único
     *
     * @param  Page     $page Página inyectada desde vista
     * @param  string   $lang Etiqueta del lenguaje a usar
     * @return Response       Dirige a vista de página específica
     */
    public function show(Page $page, $lang = 'en')
    {
        /* Buscar la autorización */
        $this->authorize('update', $page);

        /* Antes de cualquier cosa, cambiar el lenguaje */
        app()->setLocale($lang);

        /* Recuperar los mediafiles si es que es una petición JSON */
        if (\Request::ajax()) {
            return response()->json($page->mediafiles);
        }

        /* Obtener datos para usos futuros */
        $room = $page->room;
        $special = $page->special;
        // $call_to_action = json_decode($special->cta, true);

        /* Recuperar variables comunes */
        $returnData = $this->getPageCommonData($page);

        /* Incluir variables específicas */
        $returnData = $returnData->merge([
            'lang' => $lang,
            'alter_lang' => ($lang == 'en') ? 'es' : 'en',
            'services' => $this->serviceRepository->list(),
            'categories' => $this->pageRepository->getCategories(),
            'amenities' => $this->pageRepository->getAmenities(),
            'room' => $room,
            'special' => $special,
            'room_amenities' => (!is_null($room)) ? $room->amenities->pluck('id') : collect(),
            'list_pages' => ($page->pageRelations->isNotEmpty()) ? $this->pageRepository->listWithPropertyTranslations($page->site)->pluck('name', 'id') : collect(),
        ]);

        /* Retornar vista (segundo parámetro DEBE ser arreglo) */
        return view('MGCms2019::admin.pages.show', $returnData->all());
    }

    /**
     * Método público de actualización de página
     *
     * NOTA: Validación en UpdatePageRequest@rules
     *
     * @param  Page                 $page     Página inyectada desde vista
     * @param  UpdatePageRequest    $request  Información a actualizar
     * @return Response|boolean               Dirige a vista de página específica
     */
    public function update(Page $page, UpdatePageRequest $request)
    {
        $messageSuccess = 'Actualización de información general correcta.';
        $messageFailure = 'Ups. Algo salió mal con la actualización información general';

        /* Buscar la autorización */
        $this->authorize('update', $page);

        /** @var Page Página modificada */
        $success = $this->pageRepository->update($request, $page);

        /**
         * Bandera de indicación si el usuario puede acceder al listado
         * @var boolean
         */
        $canViewAny = Auth::user()->can('viewAny', [Page::class, $page->site]);

        /**
         * Bandera de indicación si la página pertenece al sitio actual
         * @var boolean
         */
        $isCurrentSite = ($page->site->id == $this->siteRepository->current()->id);

        /**
         * Información de ruta para los casos en que el usuario tenga o no
         * permiso para acceder al listado
         * @var string|mixed[]
         */
        $routeRetorno = ($canViewAny)
            ? ($isCurrentSite)
            ? 'pages.index-from-current'
            : ['pages.index', [$page->site]] /* Establecer respuesta de retorno. Si página pertenece a sitio actual, esconder URL */
            : ['pages.show', [$page->id, app()->getLocale()]]; /* page.show está usado para editar, por lo que comparten el mismo permiso y no es necesario revisar */

        if ($success) {
            if ($request->wantsJson()) {
                return $success;
            }

            return $this->responseController(true, $routeRetorno, $messageSuccess);
        } else if ($request->wantsJson()) {
            return response()->json($messageFailure, 500);
        } else if ($canViewAny) {
            /* Establecer respuesta de retorno. Si página pertenece a sitio actual, esconder URL */
            return $this->responseController(true, $routeRetorno, $messageFailure);
        }

        /* No resultó. Retornar redirección back() con mensaje */
        return $this->responseController(false, null, $messageFailure);
    }

    /**
     * Actualización de información de SEO
     *
     * NOTA: Validación en UpdateSeoPageRequest@rules
     *
     * @param  Page                 $page     Página inyectada desde vista
     * @param  UpdateSeoPageRequest $request  Información a actualizar
     * @return Response                       Dirige a vista de página específica
     */
    public function updateSeo(Page $page, UpdateSeoPageRequest $request)
    {
        $messageSuccess = 'Actualización de SEO correcta.';
        $messageFailure = 'Ups. Algo salió mal con la actualización de SEO.';

        /* Buscar la autorización */
        $this->authorize('update', $page);


        /**
         * Bandera de indicación si el usuario puede acceder al listado
         * @var boolean
         */
        $canViewAny = Auth::user()->can('viewAny', [Page::class, $page->site]);

        /**
         * Bandera de indicación si la página pertenece al sitio actual
         * @var boolean
         */
        $isCurrentSite = ($page->site->id == $this->siteRepository->current()->id);

        /**
         * Información de ruta para los casos en que el usuario tenga o no
         * permiso para acceder al listado
         * @var string|mixed[]
         */
        $routeRetorno = ($canViewAny)
            ? ($isCurrentSite)
            ? 'pages.index-from-current'
            : ['pages.index', [$page->site]] /* Establecer respuesta de retorno. Si página pertenece a sitio actual, esconder URL */
            : ['pages.show', [$page->id, app()->getLocale()]]; /* page.show está usado para editar, por lo que comparten el mismo permiso y no es necesario revisar */

        /* Intentar actualizar */
        if ($this->pageRepository->updateSeo($request, $page)) {
            if ($request->wantsJson()) {
                /* Actualización correcta */
                return response()->json($messageSuccess, 200);
            }

            return $this->responseController(true, $routeRetorno, $messageSuccess);
        } else if ($request->wantsJson()) {
            return response()->json($messageFailure, 500);
        } else if ($canViewAny) {
            return $this->responseController(true, $routeRetorno, $messageFailure);
        }

        /* No resultó. Retornar redirección back() */
        return $this->responseController(false, null, $messageFailure);
    }

    /**
     * Método público que actualiza la jerarquía de páginas del sitio solicitado
     *
     * No es necesario enviar el sitio porque la jerarquía está basada en los IDs de habitaciones
     *
     * NOTA: Validación en UpdateTreePageRequest@rules
     *
     * @param  UpdateTreePageRequest $request Información de actualización
     * @return void
     */
    public function updateTree(UpdateTreePageRequest $request)
    {
        /* Datos parseados para uso futuro */
        $data = json_decode(
            $request->getContent(),
            true /* Convertir a arreglo asociativo (con etiquetas) */
        );

        if (!empty($data)) {
            /* Buscar la autorización */
            $this->authorize('updateFromSite', [Page::class, Page::find(head($data)['id'])->site]);

            /* Ejecutar actualización */
            $this->pageRepository->rebuildTree($data);
        }
    }

    /**
     * Método público de carga de archivos para página específica
     *
     * NOTA: Validación en UploadFilePageRequest@rules
     *
     * @param  Request        $request Información del archivo a guardar
     * @return Response                Información con colección del resultado
     *                                 o código de estado HTTP
     */
    public function upload(UploadFilePageRequest $request)
    {
        /* Construir ruta de directorio para usos futuros */
        $uploadPath = 'mediafiles/pages/' . date('Y') . '/' . date('m') . '/' . $request->page;

        /* Establecer lenguaje */
        app()->setLocale($request->lang);

        try {
            /* Buscar las páginas o arrojar excepción */
            $page = $this->pageRepository->findOrFail($request->page);
        } catch (Exception $e) {
            /* Página no fue encontrada. Retornar */
            return response()->json(null, 404);
        }

        /* Buscar las autorizaciones */
        $this->authorize('update', $page);
        $this->authorize('create', [Mediafile::class, $page->site]);

        /* Añadir dirección a la petición */
        $request->request->add(['path' => $uploadPath]);

        try {
            /* Guardar archivo, crear registro y asignar a la página */
            $mediafile = $this->mediafileRepository->createWithFileByRequest(
                $request,
                null, /* Espacio de disco predeterminado */
                $page
            )->load('translations');
        } catch (ModelNotFoundException $e) {
            /* Modelos no cueron encontrados. Retornar */
            return response()->json(null, 404);
        } catch (Exception $e) {
            return response()->json($e->getMessage(), 500);
        }

        /* 201: Created */
        return response(
            collect([
                'model' => $mediafile,
                'route' => route('page.get.file', $mediafile),
                'delete' => route('mediafiles.destroy', $mediafile),
                'update' => route('files.update'),
                'lang' => $request->lang,
                'lang-delete-routes' => collect(
                    app()->make('translatable.locales')->all()
                )->mapWithKeys(function ($locale) use ($mediafile) {
                    return ["{$locale}" => route('mediafiles.translations.destroy', [$mediafile, $locale])];
                })->all()
            ]),
            201
        );
    }

    /**
     * Método público de actualización de permalinks
     *
     * @param  Site        $site    Sitio inyectado desde vista
     * @return Response             Redirección a página anterior, con mensaje
     */
    public function updatePermaLinks(Site $site)
    {
        /* Establecer sitio a tratar */
        if (!$site->exists) {
            $site = $this->siteRepository->current();
        }

        if (is_null($site)) {
            /* No existen sitios definidos */
            return $this->responseNoSitesFound();
        }

        /* Buscar la autorización */
        $this->authorize('updateFromSite', [Page::class, $site]);

        $this->pageRepository->updatePermaLinks($site);

        /* false efectura un redirect()->back() */
        return $this->responseController(false, null, 'Actualización de enlaces permanentes ');
    }

    /**
     * Método público que retorna un objeto Mediafile (usado con Eager Loading).
     *
     * El retorno es parseado a JSON y, al acceder a la ruta, retorna Response
     *
     * @param  Mediafile  $mediafile  El objeto cargado por eager loading
     * @param  string     $lang       Identificador de internacionalización
     *                                ("es", "en")...
     * @return Mediafile              Objeto Mediafile inyectado
     */
    public function getMedia(Mediafile $mediafile, $lang = 'en')
    {
        /* Buscar la autorización */
        $this->authorize('view', $mediafile);

        /* Establecer la aplicación de acuerdo con el lenguaje */
        app()->setLocale($lang);

        return $mediafile->load('translations');
    }

    /**
     * Método público que elimina un objeto Mediafile (usado con Eager Loading).
     *
     * @return Response Redirección a página anterior
     */
    public function deleteMedia(Mediafile $mediafile)
    {
        /* Buscar la autorización */
        $this->authorize('delete', $mediafile);

        $mediafile->delete();
        return redirect()->back();
    }

    /**
     * Método público que destruye la página enviada, junto con sus dependencias
     *
     * @param  Page     $page Página inyectada desde vista
     * @return Response       Dirige a vista que solicitó el cambio
     */
    public function destroy(Page $page)
    {
        /* Buscar la autorización */
        $this->authorize('delete', $page);

        $this->pageRepository->deleteAll($page);
        return redirect()->back();
    }

    /**
     * Método público que retorna jerarquía en colección.
     *
     * El retorno es parseado a JSON y, al acceder a la ruta, retorna Response
     * @param  Site                         $site    Sitio inyectado desde vista
     * @return Kalnoy\Nestedset\Collection           Jerarquía completa de páginas
     */
    public function list(Site $site)
    {
        /* Establecer el sitio a trabajar */
        if (!$site->exists) {
            $site = $this->siteRepository->current();
        }

        if (is_null($site)) {
            /* No existen sitios definidos */
            return $this->responseNoSitesFound();
        }

        /* Buscar la autorización */
        $this->authorize('viewAny', [Page::class, $site]);

        return $this->pageRepository->toTrees($site);
    }

    /**
     * Método público para retornar todos los permalinks en raíz del sitio
     *
     * @param  Site     $site    Sitio inyectado desde vista
     * @return Response          Objeto JSON con los datos
     */
    public function permalinks(Site $site)
    {
        /* Establecer el sitio a trabajar */
        if (!$site->exists) {
            $site = $this->siteRepository->current();
        }

        if (is_null($site)) {
            /* No existen sitios definidos */
            return $this->responseNoSitesFound();
        }

        /* Buscar la autorización */
        $this->authorize('viewFromSite', [Page::class, $site]);

        return response()->json($this->pageRepository->toTrees($site)->pluck('perman_link'));
    }

    /**
     * Método público pra establecer el lenguaje de la aplicación
     *
     * @param  string     $lang identificador de localización (Ej.: "en", "es"...)
     * @return Response         Dirige a vista que solicitó el cambio
     */
    public function language($lang)
    {
        app()->setLocale($lang);
        return redirect()->back();
    }

    /**
     * Método público para eliminar la traducción de una página
     *
     * @param  Page       $page   Página que tiene la traducción asociada
     * @param  string     $lang   Identificador de localización
     *                            (Ej.: "en", "es"...)
     * @return Response           Retorno al índice de páginas
     */
    public function destroyTranslation(Page $page, $lang)
    {
        /* Buscar la autorización */
        $this->authorize('update', $page);

        /** @var Collection|string[] Las localizaciones de la página */
        $translations = $page->translations()->pluck('locale');

        /* Sólo eliminar traducción si existen más */
        $allow_destruction = $translations->count() > 1;

        if ($allow_destruction) {
            $page->deleteTranslations($lang);

            /* Quitar el lenguaje de los disponibles */
            $translations = $translations->diff([$lang]);

            /* Actualizar permalinks */
            $this->pageRepository->updatePermaLinks($page->site);
        }

        /* Verificar que pueda ir al listado */
        if (Auth::user()->can('viewAny', [Page::class, $page->site])) {
            /* Establecer respuesta de retorno. Si página pertenece a sitio actual, esconder URL */
            $retorno = ($page->site->id == $this->siteRepository->current()->id)
                ? redirect()->route('pages.index-from-current')
                : redirect()->route('pages.index', $page->site);
        } else {
            $local_language = app()->getLocale();

            /* Establecer el lenguaje de retorno */
            $return_language = $translations->contains($local_language)
                ? $local_language
                : $translations->first();

            /* Retornar a la página */
            $retorno = redirect()->route('pages.show', [
                $page,
                ($allow_destruction) ? $return_language : $lang
            ]);
        }

        if (!$allow_destruction) {
            /* La página debe tener al menos una traducción. Solicitud no puede ser ejecutada */
            return $retorno->withSuccess('No es posible eliminar. Cree una nueva traducción antes o elimine completamente la página.');
        }

        /* La traducción sí fue permitida */
        return $retorno->withSuccess('Traducción eliminada exitosamente.');
    }

    /**
     * Método privado para recuperar la información pertinente a una página (para uso en las vistas)
     *
     * @param  Page                 $page El objeto Page
     * @return Collection|mixed[]         Arreglo con etiquetas
     *                                    con el nombre de la variable
     */
    private function getPageCommonData(Page $page)
    {
        /* Cargar los modelos faltantes */
        $page->loadMissing(['translations', 'cards.translations', 'mediafiles.type', 'galleries', 'pageRelations', 'galleries.items.mediafile', 'site']);

        /* Retornar datos */
        return collect([
            'page' => $page,
            'layouts' => $this->pageRepository->getLayouts(),
            'templates' => $this->pageRepository->getTemplates(),
            'forms' => $this->formRepository->list(),
            'files' => $this->filetypeRepository->list(),
        ]);
    }
}
