<?php

namespace AvengersMG\MGCms2019\App\Http\Controllers\Backend\Mediafiles;

use \Exception;
use AvengersMG\MGCms2019\App\Cms\Mediafiles\Mediafile;
use AvengersMG\MGCms2019\App\Cms\Mediafiles\Repositories\FiletypesInterface;
use AvengersMG\MGCms2019\App\Cms\Mediafiles\Repositories\MediafilesInterface;
use AvengersMG\MGCms2019\App\Cms\Mediafiles\Request\CreateDirectoryRequest;
use AvengersMG\MGCms2019\App\Cms\Mediafiles\Request\UpdateMediafileRequest;
use AvengersMG\MGCms2019\App\Cms\Mediafiles\Request\UploadFileMediafileRequest;
use AvengersMG\MGCms2019\App\Cms\Sites\Repositories\SiteInterface;
use AvengersMG\MGCms2019\App\Cms\Sites\Site;
use AvengersMG\MGCms2019\App\Http\Controllers\Controller;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class MediaFilesController extends Controller
{
    /** @var MediafilesInterface Repositorio de objetos Mediafile */
    protected $media;

    /** @var FiletypesInterface Repositorio de objetos FileType */
    protected $filetype;

    /** @var SiteInterface Repositorio de objetos Site */
    protected $site;

    /** @var string Nombre del directoroio principal */
    protected $mainDir = 'mediafiles';

    /**
     * Constructor de controlador
     *
     * @param  MediafilesInterface $mediafiles Repositorio de objetos Mediafile
     * @param  FiletypesInterface  $filetypes  Repositorio de objetos FileType
     * @param  SiteInterface       $sites      Repositorio de objetos Site
     * @return void
     */
    public function __construct(
        MediafilesInterface $mediafiles,
        FiletypesInterface  $filetypes,
        SiteInterface  $sites
    ) {
        /* Provisionar el controlador */
        $this->media = $mediafiles;
        $this->filetype = $filetypes;
        $this->site = $sites;
    }

    /**
     * Método público de listado de objetos
     *
     * @param  Site     $site Objeto Site inyectado desde la vista
     * @return Response       Vista de listado de objetos Mediafile
     */
    public function index(Site $site)
    {
        /* Buscar sitio actual de acuerdo con el dominio */
        if (!$site->exists) {
            $site = $this->site->current();
        }

        if (is_null($site)) {
            /* No existen sitios definidos */
            return $this->responseNoSitesFound();
        }

        /* Buscar la autorización */
        $this->authorize('viewAny', [Mediafile::class, $site]);

        /* Especificar directorio base */
        $base_path = $this->mainDir;

        /* Obtener todos los tipos de archivo desde base de datos */
        $files_types = $this->filetype->list();

        /* Recuperar lista de archivos desde directorio */
        $files = $this->media->listfiles($this->mainDir, $site);

        /* Retornar dirección para usos en frente */
        $path = '';

        /* Crear valor público de dirección */
        $public_path = $base_path;

        /* Retornar vista */
        return view('MGCms2019::admin.mediafiles.index', compact('site', 'files', 'base_path', 'public_path', 'path', 'files_types'));
    }

    /**
     * Método público de guardado de archivo de medio
     *
     * NOTA: Validación en UploadFileMediafileRequest@rules
     *
     * @param  UploadFileMediafileRequest    $request Información de solicitud
     * @return Response                               Información de objeto
     *                                                creado o estado HTTP
     */
    public function store(UploadFileMediafileRequest $request)
    {
        /* Sitio asociado */
        if ($request->has('site_id')) {
            $site = Site::find($request->site_id);
        } else {
            $site = Site::current();
            $request->request->add(['site_id' => $site->id]);
        }

        if (is_null($site)) {
            /* No existen sitios definidos */
            return $this->responseNoSitesFound();
        }

        /* Buscar la autorización */
        $this->authorize('create', [Mediafile::class, $site]);

        try {
            /* Guardar archivo, crear registro y asignar al sitio */
            $mediafile = $this->media->createWithFileByRequest(
                $request,
                null, /* Espacio de disco predeterminado */
                $site
            )->load(['translations', 'type']);
        } catch (ModelNotFoundException $e) {
            /* Sitio no fue encontrado. Retornar */
            return response()->json(null, 404);
        } catch (Exception $e) {
            /* Error. Retornar */
            return response()->json($e->getMessage(), 500);
        }

        /* Todo terminó correctamente. Retornar la respuesta */
        return response(
            collect([
                'model' => $mediafile,
                'route' => route('mediafiles.show', $mediafile),
                'delete' => route('mediafiles.destroy', $mediafile),
                'update' => route('mediafiles.update', $mediafile),
                'lang' => app()->getLocale(),
                'lang-delete-routes' => collect(
                    app()->make('translatable.locales')->all()
                )->mapWithKeys(function($locale) use ($mediafile){
                    return ["{$locale}" => route('mediafiles.translations.destroy', [$mediafile, $locale])];
                })->all()
            ]),
            201 /* 201: Created */
        );
    }

    /**
     * Método público para retornar el Mediafile solicitado
     * @param  Mediafile          $mediafile   Objeto cargado con Eager Loading
     * @param  string             $lang        Identificador de localización
     *                                          ("es", "en")
     * @return Response|Mediafile              Respuesta interna de sistema
     */
    public function show(Mediafile $mediafile, $lang = "en")
    {
        /* Buscar la autorización */
        $this->authorize('view', $mediafile);

        /* Establecer la aplicación de acuerdo con el lenguaje */
        app()->setLocale($lang);

        return $mediafile->load(['translations', 'type']);
    }

    /**
     * Método público para la actualización de modelos MediaFile
     *
     * NOTA: Validación en UpdateMediafileRequest@rules
     *
     * @param  UpdateMediafileRequest $request    Información de la solicitud
     * @param  Mediafile              $mediafile  Objeto cargado con Eager Loading
     * @return Response|MediaFile                 Respuesta en JSON sobre las funciones
     */
    public function update(UpdateMediafileRequest $request, Mediafile $mediafile)
    {
        /* Buscar la autorización */
        $this->authorize('update', $mediafile);

        try {
            $file = $this->media->updateAndRenameFileByRequest($request, $mediafile);
        } catch (ModelNotFoundException $e) {
            /* Mediafile no fue encontrado */
            return response()->json(null, 404);
        } catch (Exception $e) {
            /* Mediafile no fue actualizado. Retornar error*/
            return response()->json($e->getMessage(), 500);
        }

        /* Todo terminó correctamente. Retornar respuesta */
        return $file->load(['translations', 'type']);
    }

    /**
     * Método público para eliminar un modelo MediaFile
     *
     * @param  Mediafile  $mediafile Objeto cargado con Eager Loading
     * @return Response              Redirección a la página solicitante
     *                               o estado de petición
     */
    public function destroy(Mediafile $mediafile)
    {
        /* Buscar la autorización */
        $this->authorize('delete', $mediafile);

        try {
            /* Intentar eliminar */
            $this->media->deleteAndRemoveFile($mediafile);
        } catch (Exception $e) {
            /* Error. Redirigir a página */
            return response($e->getMessage(), 500);
        }

        /* Si es AJAX, retornar estado satisfactorio, sin contenido */
        if (request()->ajax()) {
            /* 204: No Content */
            return response(null, 204);
        }

        /* Todo finalizó correctamente. Retornar a página */
        return redirect()->back();
    }

    /**
     * Método público para eliminar una traducción de un modelo MediaFile
     *
     * @param  Mediafile  $mediafile Objeto cargado con Eager Loading
     * @param  string     $lang      Identificador de localización
     *                               (Ej. "en", "es"...)
     * @return Response              Respuesta con el objeto actualizado
     *                               o estado HTTP
     */
    public function destroyTranslation(Mediafile $mediafile, $lang = 'en')
    {
        /* Buscar la autorización */
        $this->authorize('update', $mediafile);

        /* Traducción a eliminar */
        $translation = $mediafile->translate($lang);

        if (!is_null($translation)) {
            /* Cantidad de traducciones */
            $translations_quantity = $mediafile->translations()->count();

            /* Sólo eliminar si quedaría al menos una */
            if ($translations_quantity > 1) {
                /* Traducción existe. Borrar */
                $translation->delete();
            } else {
                /*
                  409 - Conflict: No es posible eliminar la única
                  traducción disponible
                 */
                return abort(409);
            }
        }

        /* Si es AJAX, retornar el modelo con las traducciones actualizadas */
        if (request()->ajax()) {
            return $mediafile->load(['translations', 'type']);
        }

        /* Retornar a la página anterior */
        return redirect()->back();
    }

    /**
     * Método público para crear un nuevo directorio
     *
     * NOTA: Validación en CreateDirectoryRequest@rules
     *
     * @param  CreateDirectoryRequest $request Información de solicitud
     * @return Response                        Redirección a página solicitante
     *                                         o retornar dirección
     */
    public function createDir(CreateDirectoryRequest $request)
    {
        /* Buscar la autorización */
        $this->authorize('createDirectory', Mediafile::class);

        /* Crear directorio */
        $this->media->createDirectoryByRequest($request);

        /* Si es AJAX, retornar el código correcto */
        if ($request->ajax()) {
            /* 201: Created */
            return response($request->path.'/'.$request->name, 201);
        }

        /* Regresar a página anterior */
        return redirect()->back();
    }

    /**
     * Método público para mostrar los archivos en un directorio
     *
     * @param  Site     $site  Objeto Site cargado por Eager Loading
     * @param  string   $dir   URI de directorio
     * @return Response        Vista de listado de archivos
     *                         o estado de la petición
     */
    public function showDir(Site $site, $dir)
    {
        /* Buscar sitio actual de acuerdo con el dominio */
        if (!$site->exists) {
            $site = $this->site->current();
        }

        if (is_null($site)) {
            /* No existen sitios definidos */
            return $this->responseNoSitesFound();
        }

        /* Buscar la autorización */
        $this->authorize('viewAny', [Mediafile::class, $site]);

        /* Obtener dirección de directorio, quitando la diagonal extra si la hay */
        $path = rtrim($dir, '/');

        /* Obtener todos los tipos de archivos */
        $files_types = $this->filetype->list();

        /* Especificar directorio base */
        $base_path = $this->mainDir;

        /* Crear valor público de dirección */
        $public_path = (empty($path))? $base_path : implode('/', [$base_path, $path]);

        /* Construir breadcrumbs con el URI, quitando la diagonal extra si existe */
        $breadcrumbs = explode('/', rtrim($dir, '/'));

        /* Construir el directorio local */
        $localDir = implode('/', [
              $this->mainDir,
              $path,
        ]);

        /* Verificar que directorio local existe */
        if ($this->media->isDirectory($localDir)) {
            /* Recuperar directorios y archivos */
            $files = $this->media->listfiles($localDir, $site);

            /* Redirigir a la vista junto con las variables */
            return view('MGCms2019::admin.mediafiles.index', compact('site', 'files', 'base_path', 'public_path', 'path', 'files_types', 'breadcrumbs'));
        }

        /* No es directorio. Detener */
        return abort(404);
    }

    /**
     * Método público para actualizar un objeto Mediafile de acuerdo
     * con la solicitud
     *
     * Delegado desde MediaController y refactorizado
     *
     * @param  Request $request Petición con los datos a actualizar
     * @return Response         El modelo Mediafile, serializado
     *                          o estado de la petición
     */
    public function fileUpdate(Request $request)
    {
        /* Mediafile asociado */
        $mediafile = $this->media->find($request->id);

        /* Buscar la autorización */
        $this->authorize('update', $mediafile);

        if ($request->ajax()) {
            try {
                return $this->media->updateAndRenameFileByRequest($request, $mediafile)->loadMissing(['translations', 'type']);
            } catch (Exception $e) {
                /* Error. Redirigir a página */
                return abort(500);
            }
        }
    }
}
