<?php

namespace AvengersMG\MGCms2019\App\Http\Controllers\Backend\Mediafiles;

use \Exception;
use AvengersMG\MGCms2019\App\Cms\Mediafiles\FileType;
use AvengersMG\MGCms2019\App\Cms\Mediafiles\Mediafile;
use AvengersMG\MGCms2019\App\Cms\Mediafiles\Repositories\FiletypesInterface;
use AvengersMG\MGCms2019\App\Cms\Mediafiles\Request\CreateFiletypeRequest;
use AvengersMG\MGCms2019\App\Cms\Mediafiles\Request\UpdateFiletypeRequest;
use AvengersMG\MGCms2019\App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class FileTypesController extends Controller
{
    /** @var FiletypesInterface Repositorio de objetos FileType */
    protected $filetype;

    /**
     * Constructor de controlador
     *
     * @param  FiletypesInterface  $filetypes  Repositorio de objetos FileType
     * @return void
     */
    public function __construct(
        FiletypesInterface  $filetypes
    ) {
        /* Provisionar el controlador */
        $this->filetype = $filetypes;
    }

    /**
     * Método público para retornar la lista de objetos FileType
     *
     * @return Response Vista de listado
     */
    public function index()
    {
        /* Buscar la autorización */
        $this->authorize('viewAny', FileType::class);

        $filetypes = $this->filetype->list();
        return view('MGCms2019::admin.filestype.index', compact('filetypes'));
    }

    /**
     * Método público para mostrar la vista de creación de objeto FileType
     *
     * @return Response Vista de creación
     */
    public function create()
    {
        /* Buscar la autorización */
        $this->authorize('create', FileType::class);

        return view('MGCms2019::admin.filestype.create');
    }

    /**
     * Método público para la creación de un nuevo objeto FileType de acuerdo con solicitud
     *
     * NOTA: Validación en CreateFiletypeRequest@rules
     *
     * @param  CreateFiletypeRequest  $request Información de solicitud
     * @return Response                        Redirección a vista de listado
     */
    public function store(CreateFiletypeRequest $request)
    {
        $return_route = 'filetypes.index';

        /* Buscar la autorización */
        $this->authorize('create', FileType::class);

        $this->filetype->create($request->all());

        /* Si el usuario no tiene permiso de listar tipos de archivo, retornar a creación */
        if (Auth::user()->cant('viewAny', FileType::class)) {
            $return_route = 'filetypes.create';
        }

        return redirect()->route($return_route);
    }

    /**
     * Método público para mostrar la vista de edición de objeto FileType
     *
     * @param  FileType $filetype Objeto cargado con Eager Loading
     * @return Response           Vista de edición
     */
    public function edit(FileType $filetype)
    {
        /* Buscar la autorización */
        $this->authorize('update', $filetype);

        return view('MGCms2019::admin.filestype.edit')->with('file', $filetype);
    }

    /**
     * Método público para la edición de un objeto FileType de acuerdo con solicitud
     *
     * NOTA: Validación en UpdateFiletypeRequest@rules
     *
     * @param  UpdateFiletypeRequest  $request    Información de solicitud
     * @param  FileType               $filetype   Objeto cargado con Eager Loading
     * @return Response                           Redirección de acuerdo con resultado
     */
    public function update(UpdateFiletypeRequest $request, FileType $filetype)
    {
        /* Buscar la autorización */
        $this->authorize('update', $filetype);

        try {
            $filetype->update($request->all());
        } catch (Exception $e) {
            /* Error general */
            return abort(500);
        }

        /* Si el usuario no tiene permiso de listar tipos de archivo, retornar a edición */
        if (Auth::user()->cant('viewAny', FileType::class)) {
            return redirect()->route('filetypes.edit', $filetype->id);
        }

        /* Todo terminó correctamente. Retornar */
        return redirect()->route('filetypes.index');
    }

    /**
     * Método público para la destrucción de un objeto FileType
     *
     * @param  FileType $filetype Objeto cargado con Eager Loading
     * @return Response           Redirección de acuerdo con resultado
     */
    public function destroy(FileType $filetype)
    {
        /* Buscar la autorización */
        $this->authorize('delete', $filetype);

        try {
            $filetype->delete();
        } catch (Exception $e) {
            /* Error general */
            return abort(500);
        }

        /* Todo terminó correctamente. Retornar */
        return redirect()->back();
    }
}
