<?php

namespace AvengersMG\MGCms2019\App\Http\Controllers\Backend\Components;

use AvengersMG\MGCms2019\App\Cms\Components\Galleries\Gallery;
use AvengersMG\MGCms2019\App\Cms\Components\Galleries\Repositories\GalleryInterface;
use AvengersMG\MGCms2019\App\Cms\Components\Galleries\Requests\UpdateGalleryRequest;
use AvengersMG\MGCms2019\App\Cms\Mediafiles\Repositories\MediafilesInterface;
use AvengersMG\MGCms2019\App\Cms\Pages\Page;
use AvengersMG\MGCms2019\App\Cms\Pages\Repositories\PageInterface;
use AvengersMG\MGCms2019\App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class GalleryController extends Controller
{
    /** @var galleryRepository Repositorio de objetos Page */
    protected $galleryRepository;

    /** @var PageInterface Repositorio de objetos Page */
    private $pageRepository;

    /** @var MediafilesInterface Repositorio de objetos Mediafile */
    private $mediafileRepository;

    /**
     * Constructor de controlador
     *
     * @param PageInterface         $pageInterface       Repositorio de
     *                                                   modelo Page
     * @param GalleryInterface      $galleryInterface    Repositorio de
     *                                                   modelo Gallery
     * @param MediafilesInterface   $mediafileInterface  Repositorio de
     *                                                   modelo Mediafile
     */
    public function __construct(
        PageInterface $pageInterface,
        GalleryInterface $galleryInterface,
        MediafilesInterface $mediafileInterface
    ) {
        /* Provisionar controlador con los repositorios */
        $this->galleryRepository = $galleryInterface;
        $this->pageRepository = $pageInterface;
        $this->mediafileRepository = $mediafileInterface;
    }

    /**
     * @var page_id Vinculacion implicita del model page
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function create($page_id)
    {
        /** Busca la pagina a la que se le va a agregar la galeria */
        $page = $this->pageRepository->findOrFail($page_id);

        /* Buscar la autorización */
        $this->authorize('update', $page);

        return view('MGCms2019::admin.components.galleries.create', compact('page'));
    }

    /**
     * @param  \Illuminate\Http\Request  $request
     * @return instance of Illuminate\Routing\Redirector
     */
    public function store(Request $request)
    {
        /** validamos los datos que recibimos */
        $request->validate([
            'name' => 'required',
            'images' => 'sometimes',
        ]);

        $page = $this->galleryRepository->create($request->all());

        /* Buscar la autorización */
        $this->authorize('update', $page);

        /** retorna a la vista de la pagina a la que se le agrego la galeria */
        return redirect()->route('pages.show', [$page, app()->getLocale()]);
    }

    /**
     * @param gallery int - Vinculacion automatica del model Gallery
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function edit(Gallery $gallery)
    {
        /* Buscar la autorización */
        $this->authorize('update', $gallery->page);

        /**
         * (usado con Eager Loading)
         * Obtenemos los items y filtramos solo por mediafile_id
         * despues lo convertimos en array
         */
        $arr_images = $gallery->items->pluck('mediafile_id')->toArray();

        /** Obtenemos los archivos multimedia que pertenecen a una pagina
         * (usado con Eager Loading)
         */
        $page_mediafiles = $gallery->page->mediafiles;

        /* Obtener los IDs de los de la página */
        $page_mediafiles_ids = $page_mediafiles->pluck('id')->all();

        /** @var Collection|Mediafile[] Lista de mediafiles adicionales */
        $extra_mediafiles = $this->mediafileRepository->find(
            collect($arr_images)->diff($page_mediafiles_ids)->all()
        );

        /*
          Lista de todos los mediafiles a mostrar, incluyendo los escogidos
          que no están asociados directamente a la página
         */
        $mediafiles = $page_mediafiles->merge($extra_mediafiles);

        return view('MGCms2019::admin.components.galleries.edit', compact('gallery', 'mediafiles', 'arr_images'));
    }

    /**
     * Método que controla la prioridad de los ítems en las galerías.
     *
     * @return instance of Illuminate\Routing\Redirector
     */
    public function update(Request $request, Gallery $gallery)
    {
        /* Buscar la autorización */
        $this->authorize('update', $gallery->page);

        if ($request->ajax()) {
            $response = $this->galleryRepository->prepareToUpdate(
                json_decode(
                    $request->getContent() /* Cadena que representa un arreglo */
                ),
                $gallery
            );
            return response()->json($response);
        }
    }

    /**
     * Metodo para actualizar las fotos de una galeria
     * @param UpdateGalleryRequest  $request
     * @param gallery Vinculacion implicita del modelo Gallery
     * @return instance of Illuminate\Routing\Redirector
     */
    public function updateGallery(UpdateGalleryRequest $request, Gallery $gallery)
    {
        $messageSuccess = 'Actualización de galería correcta.';
        $messageFailure = 'Ups. Algo salió mal con la actualización de galería.';

        /* Página asociada */
        $page = $gallery->page;

        /* Buscar la autorización */
        $this->authorize('update', $page);

        /*
            ADVERTENCIA: El método updateOrInsert() en el repositorio
            no genera retorno, por lo que no es posible avisar de fallas
         */

        // LLamado a un metodo en el repositorio.
        $this->galleryRepository->updateOrInsert($request, $gallery);

        if ($request->wantsJson()) {
            /* Retornar objeto actualizado */
            return $gallery->fresh();
        }

        /* La petición no solicita JSON: Retornar a la página  */
        return $this->responseController(true, ['pages.show', [$page->id, app()->getLocale()]], $messageSuccess);
    }

    /**
     * @param int id ID en la tabla de galleries
     */
    public function destroy($id)
    {
        /** Busca el registro en la BD */
        $gallery = $this->galleryRepository->findOrFail($id);

        /* Página asociada */
        $page = $gallery->page;

        /* Buscar la autorización */
        $this->authorize('update', $page);

        /* Eliminar galería. Los ítems asociados son eliminados por cascadeo */
        $gallery->delete();

        /* Redirigir a la página asociada */
        return redirect()->route('pages.show', [$page, app()->getLocale()]);
    }


    /**
     *
     */
    public function reorder(Gallery $gallery)
    {
        /* Buscar la autorización */
        $this->authorize('update', $gallery->page);

        return view('MGCms2019::admin.components.galleries.reorder', compact('gallery'));
    }

    /**
     * Método público para recuperar los objetos mediafiles que sean del sitio
     * pero no estén asociados a la página. Los resultados son paginados, por lo
     * que leerá la variable numérica "page" en el URI
     *
     * @param  Gallery  Objeto Gallery cargado con Eager Loading
     * @param  Page     Objeto Page cargado con Eager Loading
     * @return Response Colección de Mediafiles extras
     */
    public function extraMediafiles(Gallery $gallery, Page $page)
    {
        /* Página asociada, para uso futuro */
        if ($gallery->exists) {
            $target_page = $gallery->page;
        } elseif ($page->exists) {
            $target_page = $page;
        } else {
            /* Petición no procesable */
            return abort(422);
        }

        /* Buscar la autorización */
        $this->authorize('update', $target_page);

        /* Omitir traer los que ya están asociadas a la página */
        $page_mediafiles_ids = $target_page->mediafiles()->pluck('id');

        /* Mediafiles que la página ya tiene */
        $exclude_ids = ($gallery->exists)
            ? $gallery->items()->pluck('mediafile_id')->merge($page_mediafiles_ids)->unique()->all()
            : $page_mediafiles_ids->all();

        return $this->mediafileRepository
            ->bySite($target_page->site)
            ->imagesOnly()
            ->whereNotIn('id',  $exclude_ids)
            ->paginate(16);
    }

    /**
     * Método para recuperar
     * @param  Request  $request
     * @param  Page     $page    Página a la que las galerías pertenecen
     * @return Response          Resultado de actualización
     */
    public function reorderFromPage(Request $request, Page $page)
    {
        /* Buscar la autorización */
        $this->authorize('update', $page);

        /* Ejecutar si es AJAX */
        if ($request->ajax()) {
            $newOrder = json_decode(
                $request->getContent() /* Cadena que representa un arreglo */
            );

            /** @var boolean Resultado de operación */
            $response = $this->galleryRepository->updatePageGalleriesOrder(
                $newOrder,
                $page
            );

            if ($response) {
                return response()->json($response);
            }

            /* Error general */
            return abort(500);
        }

        /* No es AJAX. Petición no procesable */
        return abort(422);
    }
}
