<?php

namespace AvengersMG\MGCms2019\App\Http\Controllers\Backend\Components;

use AvengersMG\MGCms2019\App\Cms\Components\Cards\Card;
use AvengersMG\MGCms2019\App\Cms\Components\Cards\Repositories\CardInterface;
use AvengersMG\MGCms2019\App\Cms\Mediafiles\Repositories\MediafilesInterface;
use AvengersMG\MGCms2019\App\Cms\Pages\Page;
use AvengersMG\MGCms2019\App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class CardController extends Controller
{

    /** @var cardRepository Repositorio de objetos Card */
    protected $cardRepository;

    /** @var MediafilesInterface Repositorio de objetos Mediafile */
    private $mediafileRepository;

    /**
     * Constructor del naveegador
     * @param cardInterface         $cardInterface       Repositorio de
     *                                                   modelo Card
     * @param MediafilesInterface   $mediafileInterface  Repositorio de
     *                                                   modelo Mediafile
     */
    public function __construct(
        CardInterface $cardInterface,
        MediafilesInterface $mediafileInterface
    )
    {
        $this->cardRepository = $cardInterface;
        $this->mediafileRepository = $mediafileInterface;
    }

    /**
     * Método público para mostrar la vista de las cards que perteneces a un pagina
     * y poder reordenarlas
     * @var page Vinculacion implicita del model page
     */
    public function index(Page $page)
    {
        $cards = $page->cards->sortBy('priority');
        return view('MGCms2019::admin.components.cards.index', compact('cards', 'page'));
    }

    /**
     * @var page Vinculacion implicita del model page
     * @var lang Variable global de lenguaje
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function create(Page $page, $lang)
    {
        /* Buscar la autorización */
        $this->authorize('update', $page);

        /** @var priority se le agrega el numero de cards que tiene una pagina + 1 */
        $priority = $page->cards->count() + 1;
        return view('MGCms2019::admin.components.cards.create', compact('page', 'priority', 'lang'));
    }

    /**
     * @param  \Illuminate\Http\Request  $request
     * @return instance of Illuminate\Routing\Redirector
     */
    public function store(Request $request)
    {
        /* Página asociada */
        $page = Page::find($request->page_id);

        /* Buscar la autorización */
        $this->authorize('update', $page);

        // Create Card
        $card = $this->cardRepository->create($request->all());

        return redirect()->route('pages.show', [$page->id, $request->locale]);
    }

    /**
     * @var id ID en la tabla de Cards
     * @var lang Variable global de lenguaje
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function edit($id, $lang)
    {
        /** SETEAR variable global de lenguaje */
        \app()->setLocale($lang);

        /** Buscar la card  */
        $card = $this->cardRepository->findOrFail($id);

        /* Buscar la autorización */
        $this->authorize('update', $card->page);

        /** Obtenemos los archivos multimedia que pertenecen a una pagina
         * (usado con Eager Loading)
        */
        $page_mediafiles = $card->page->mediafiles;

        /* Obtener los IDs de los de la página */
        $page_mediafiles_ids = $page_mediafiles->pluck('id')->all();

        /*
         ADVERTENCIA: $card->image (`card_translations`.`image`) es un campo
         de texto que representa la dirección del archivo, lo que puede llevar
         a un estado inconsistente de la base de datos. Recomendado cambiar por
         ID de mediafile (Ej. `card_translations`.`mediafile_id`)
        */

        /** @var Collection|Mediafile[] Lista de mediafiles adicionales */
        $extra_mediafiles = $this->mediafileRepository->where('path', $card->image)->whereNotIn('id', $page_mediafiles_ids)->with(['translations'])->get();

        /*
          Lista de todos los mediafiles a mostrar, incluyendo los escogidos
          que no están asociados directamente a la página
         */
        $mediafiles = $page_mediafiles->merge($extra_mediafiles);

        return view('MGCms2019::admin.components.cards.edit', compact('card', 'lang', 'mediafiles'));
    }

    /**
     * Metodo para actualizar el contenido de una Card
     * @param  \Illuminate\Http\Request  $request
     * @return instance of Illuminate\Routing\Redirector
     */
    public function update(Request $request)
    {
        $messageSuccess = 'Actualización de tarjeta correcta.';
        $messageFailure = 'Ups. Algo salió mal con la actualización de tarjeta.';

        /* Página asociada */
        $page = $this->cardRepository->find($request->id)->page;

        /* Buscar la autorización */
        $this->authorize('update', $page);

        $card = $this->cardRepository->update($request->all());

        if ($card) {
            if ($request->wantsJson()) {
                return $card;
            }

            return $this->responseController(true, ['pages.show', [$page->id, $request->locale]], $messageSuccess);
        }else if ($request->wantsJson()) {
            return response()->json($messageFailure, 500);
        }

        /* No resultó. Retornar redirección back() con mensaje */
        return $this->responseController(false, null, $messageFailure);
    }

    /**
     * Metodo para eliminar una Card
     * @var id ID de un registri en la tabla Card
     * @return Illuminate\Routing\Redirector
     */
    public function destroy($id)
    {
        $card = $this->cardRepository->findOrFail($id);

        /* Buscar la autorización */
        $this->authorize('update', $card->page);

        $card->delete();
        return redirect()->back();
    }

    /**
     * Metodo que se ejecuta cuando se reordenan las Card.
     * @param  \Illuminate\Http\Request  $request
     */
    public function priority(Request $request)
    {
        /* 2019-06-19: Descodificación de JSON delegado a (este) controlador, quien es el que consume el recurso y, por tanto, encargado de formatear los datos */
        $cards = json_decode(
            $request->getContent(),
            true
        );

        /* NOTA: $cards puede no estar en el orden que indican los índices, así que head() tomará el primer valor de acuerdo con el orden original de inserción (aunque no sea el índice 0) */
        $page = $this->cardRepository->find(head($cards)['id'])->page;

        /* Buscar la autorización */
        $this->authorize('update', $page);

        if ($request->ajax()) {
            $response = $this->cardRepository->updatePriority($cards);
            return response()->json($response);
        }
    }

    /**
     * Método público para recuperar los objetos mediafiles que sean del sitio
     * pero no estén asociados a la página. Los resultados son paginados, por lo
     * que leerá la variable numérica "page" en el URI
     *
     * @param  Card     $card  Objeto Card cargado con Eager Loading
     * @param  Page     $page  Objeto Page cargado con Eager Loading
     * @param  string   $lang  Identificador de localización
     *                         (Ej. "es", "en"...)
     * @return Response Colección de Mediafiles extras
     */
    public function extraMediafiles(Card $card, Page $page, $lang = 'en')
    {
        /* Página asociada, para uso futuro */
        if ($card->exists) {
            $target_page = $card->page;
        } elseif ($page->exists) {
            $target_page = $page;
        } else {
            /* Petición no procesable */
            return abort(422);
        }

        /* Buscar la autorización */
        $this->authorize('update', $target_page);

        /* Obtener traducción de objeto Card, para uso futuro */
        $card_translation = $card->translate($lang);

        if ($card->exists && is_null($card_translation)) {
            /* El objeto no tiene la traducción solicitada */
            return abort(404);
        }

        /* Omitir traer los que ya están asociadas a la página */
        $page_mediafiles_ids = $target_page->mediafiles()->pluck('id');

        /* Mediafiles que la página ya tiene */
        $exclude_ids = ($card->exists)
            ? $this->mediafileRepository->where('path', $card_translation->image)->pluck('id')->merge($page_mediafiles_ids)->unique()->all()
            : $page_mediafiles_ids->all();

        return $this->mediafileRepository->bySite($target_page->site)->whereNotIn('id',  $exclude_ids)->paginate(16);
    }
}
