<?php

namespace AvengersMG\MGCms2019\App\Http\Controllers\Backend\Accommodations;
/* importación de clases necesarias para las acciones de RoomsController */
use AvengersMG\MGCms2019\App\Cms\Accommodations\Room;
use AvengersMG\MGCms2019\App\Cms\Accommodations\Rooms\Request\RoomRequest;
use AvengersMG\MGCms2019\App\Cms\Accommodations\Rooms\Request\UpdateRoomRequest;
use AvengersMG\MGCms2019\App\Cms\Accommodations\Rooms\RoomInterface;
use AvengersMG\MGCms2019\App\Cms\Pages\Page;
use AvengersMG\MGCms2019\App\Cms\Sites\Repositories\SiteInterface;
use AvengersMG\MGCms2019\App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class RoomsController extends Controller
{
    /* variable privada donde se almacena la instancia del la interfaz del repositorio de habitaciones */
    private $room;

    /* variable privada donde se almacena la instancia del la interfaz del repositorio de sitios */
    private $site;

    /* Variable $view que almacena la ruta de la vista de creación de habitación */
    private $view = 'MGCms2019::admin.components.accommodations.rooms.create';
    /* constructir para instanciar el repositorio de rooms en controlador para poder ser accedida desde
    cualquier método de clase */

    function __construct(
        RoomInterface $room,
        SiteInterface $site
    )
    {
        $this->room = $room;
        $this->site = $site;
    }
    /**
     * Método publico de creación de habitación
     * @param  Page   $page página inyectada desde la vista
     * @param  String $lang idioma inyecto desde la vista
     * @return Response       dirige a la vista de creación de habitación
     */
    public function create(Page $page, $lang)
    {
        /* Buscar la autorización */
        $this->authorize('update', $page);

        /* Obtiene las categorías de habitaciones */
        $categories = $this->room->getCategories();
        /* Obtine las amenidades para las habitaciones */
        $amenities = $this->room->getAmenities();

        /* retorna vista */
        return view($this->view, compact('page', 'amenities', 'categories', 'lang'));

    }

    /**
     * Método público de guardado de datos para nueva habitación
     *
     * NOTA: Validación en RoomRequest@rules
     *
     * @param  RoomRequest $request Información de habitación
     * @return Response             Dirige a vista de ver página
     */
    public function store(RoomRequest $request)
    {
        /* Encontrar la página, para usos futuros */
        $page = Page::find($request->page_id);

        /* Buscar la autorización */
        $this->authorize('update', $page);

        /* Establecer página de retorno */
        $redirect = redirect()->route('pages.show', [$page->id, $request->locale]);

        /* llama el método store del repositorio de rooms para crear el registro */
        $room = $this->room->store($request);

        if ($room === false) {
            $redirect = $redirect->withErrors("No fue posible crear la habitación");
        }

        return $redirect;
    }

    /**
     * [show description]
     * @param  Room   $room  Recibe el objeto room desde la vista
     * @param  int    $index Índice de imagen a ignorar
     * @return Response       Retorna objeto Room con la galería
     */
    public function show(Room $room, $index)
    {
        /* Buscar la autorización */
        $this->authorize('view', $room->page);

        /**
         *  Eliminando la foto
         */
        $gallery = $room->gallery;
        unset($gallery[$index]);

        /**
         * Convirtiendo a json y guardando
         */
        $gallery = json_encode($gallery);
        $room->gallery = $gallery;
        $room->save();

        return $room;
    }
    /**
     * actualizando registros de habitaciones
     * @param  UpdateRoomRequest $request Recibe los datos del formulario
     * @param  Room    $room    Contiene el modelo Room
     * @return Response         Retorna función de actualización del repositorio room
     */
    public function update(UpdateRoomRequest $request, Room $room)
    {
        $messageSuccess = 'Actualización de habitación correcta.';
        $messageFailure = 'Ups. Algo salió mal con la actualización de habitación.';

        /* Página al que está asociado, para uso futuro */
        $page = $room->page;

        /* Buscar la autorización */
        $this->authorize('update', $page);

        /**
         * Bandera de indicación si el usuario puede acceder al listado
         * @var boolean
         */
        $canViewAny = Auth::user()->can('viewAny', [Page::class, $page->site]);

        /**
         * Bandera de indicación si la página pertenece al sitio actual
         * @var boolean
         */
        $isCurrentSite = ($page->site->id == $this->site->current()->id);

        /**
         * Información de ruta para los casos en que el usuario tenga o no
         * permiso para acceder al listado
         * @var string|mixed[]
         */
        $routeRetorno = ($canViewAny)
            ? ($isCurrentSite)
                ? 'pages.index-from-current'
                : ['pages.index', [$page->site]] /* Establecer respuesta de retorno. Si página pertenece a sitio actual, esconder URL */
            : ['pages.show', [$page->id, app()->getLocale()]]; /* page.show está usado para editar, por lo que comparten el mismo permiso y no es necesario revisar */

        /** @var Room|boolean Objeto actualizado */
        $success = $this->room->prepareUpdate($request, $room);

        if ($success) {
            if ($request->wantsJson()) {
                return $success;
            }

            return $this->responseController(true, $routeRetorno, $messageSuccess);
        }else if ($request->wantsJson()) {
            return response()->json($messageFailure, 500);
        }else if($canViewAny){
            /* Establecer respuesta de retorno. Si página pertenece a sitio actual, esconder URL */
            return $this->responseController(true, $routeRetorno, $messageFailure);
        }

        /* No resultó. Retornar redirección back() con mensaje */
        return $this->responseController(false, null, $messageFailure);
    }

    /**
     * Método público para eliminar una habitación y deshacer asociaciones
     *
     * @param  Room     $room Habitación a eliminar
     * @param  string   $lang Idioma inyectado desde la página
     * @return Response       Redirección a página asociada
     */
    public function destroy(Room $room, $lang)
    {
        /* Página asociada, para uso futuro */
        $page = $room->page;

        /* Buscar la autorización */
        $this->authorize('update', $page);

        /* Eliminar modelo y traducciones */
        $this->room->deleteAll($room);

        /* Redirigir a la página que tuvo asignada */
        return redirect()->route('pages.show', [$page->id, $lang]);
    }
}
