<?php

namespace AvengersMG\MGCms2019\App\Console\Commands;

use \Exception;
use AvengersMG\MGCms2019\App\Cms\Components\Galleries\Repositories\GalleryFeatureInterface;
use AvengersMG\MGCms2019\App\Cms\Components\Galleries\Repositories\GalleryInterface;
use AvengersMG\MGCms2019\App\Cms\Components\Galleries\Repositories\ItemInterface;
use AvengersMG\MGCms2019\App\Cms\Mediafiles\Repositories\FiletypesInterface;
use AvengersMG\MGCms2019\App\Cms\Sites\Repositories\SiteInterface;
use Carbon\Carbon;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class UpdateDynamicGalleryPriorities extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'mgcms2019:update-dynamic-gallery-priorities';

    /**
     * Espacio para el repositorio de galerías
     *
     * @var GalleryInterface
     */
    protected $galleryRepository;

    /**
     * Espacio para el repositorio de fuentes de galerías
     *
     * @var GalleryFeatureInterface
     */
    protected $galleryFeatureRepository;

    /**
     * Espacio para el repositorio de ítems de galería
     *
     * @var ItemInterface
     */
    protected $itemRepository;

    /**
     * Espacio para el repositorio de sitios
     *
     * @var SiteInterface
     */
    protected $siteRepository;

    /**
     * Espacio para el repositorio de tipos de archivo
     *
     * @var FiletypesInterface
     */
     protected $filetypeRepository;


    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Añadir y eliminar imágenes dinámicamente a las galerías que lo requieran y reordenar tomando en cuenta el orden establecido por los usuarios para uso del paquete AvengersMG\\MGCms2019.';

    /**
     * Create a new command instance.
     *
     * @param GalleryInterface        $galleryInterface         Repositorio de galerías
     * @param GalleryFeatureInterface $galleryFeatureInterface  Repositorio de fuentes
     * @param ItemInterface           $itemInterface            Repositorio de ítems
     * @param SiteInterface           $siteInterface            Repositorio de sitios
     * @param FiletypesInterface      $filetypeInterface        Repositorio de tipos
     * @return void
     */
    public function __construct(
        GalleryInterface $galleryInterface,
        GalleryFeatureInterface $galleryFeatureInterface,
        ItemInterface $itemInterface,
        SiteInterface $siteInterface,
        FiletypesInterface $filetypeInterface
    )
    {
        parent::__construct();

        /* Provisionar comando */
        $this->galleryRepository = $galleryInterface;
        $this->galleryFeatureRepository = $galleryFeatureInterface;
        $this->itemRepository = $itemInterface;
        $this->siteRepository = $siteInterface;
        $this->filetypeRepository = $filetypeInterface;
    }

    /**
     * Execute the console command.
     *
     * @return mixed
     */
    public function handle()
    {
        /* Obtener la fecha de hoy, para usos múltiples */
        $now = Carbon::now();

        /* Iniciar transacción */
        DB::beginTransaction();

        try {
            $this->line('Compilando información de fuentes...');

            /** @var FileType tipo de archivo "header" */
            $headerFiletype = $this->filetypeRepository->getHeader();

            /** @var FileType tipo de archivo "header_mobile" */
            $headerMobileFiletype = $this->filetypeRepository->getHeaderMobile();

            /*
              Obtener las fuentes y las galerías a las que afectan,
              junto con su página
            */
            $allFeatures = $this->galleryFeatureRepository
                ->with(['gallery.page'])
                ->get();

            /* Obtener todos los sitios involucrados */
            $sites = $this->siteRepository->find(
                $allFeatures->pluck('gallery.page.site_id')
                    ->unique()
                    ->all()
            );

            /* Obtener todos los nombres de clases de los objetos que son fuentes */
            $featurables = $allFeatures->pluck('featurable_type')->unique();

            /**
             * Todos los datos de las fuentes
             *
             * @var Collection|Collection[]
             */
            $featureSourcesPerSiteId = $sites->mapWithKeys(function ($site) {
                return [
                    "{$site->id}" => collect(),
                ];
            });

            /*
                Ciclar por cada fuente
                (ciclo necesario por ejecución de métodos en clases externas)
            */
            $featurables->each(function ($featurable) use ($now, $headerFiletype, $headerMobileFiletype, &$featureSourcesPerSiteId) {
                /* Nombre del método que establece la relación en el objeto Page */
                $featurableMethod = call_user_func("{$featurable}::getFeaturableRelationshipMethod");

                /* Formato en el que las fechas de inicio y fin vienen */
                $featurableDateFormat = call_user_func("{$featurable}::getFeaturableDateFormat");

                /**
                  * Obtener todas las definiciones desde las páginas que tengan
                  * el tipo de fuente asociado. Después, agrupar por ID de sitio
                  *
                  * @var Collection|Collection Definiciones de fuentes, con índices
                  *                            de ID de sitio
                  */
                $currentFeatureSourcesPerSiteId = call_user_func("{$featurable}::ownerPagesQuery")
                    ->whereHas('translations', function ($query) {
                        $query->where('status', 'published');
                    }, ">", 0)
                    ->whereIn('site_id', $featureSourcesPerSiteId->keys()->all())
                    ->whereHas('mediafiles', function ($query) use ($headerFiletype, $headerMobileFiletype) {
                        /*
                            Filtrar por los que contengan las imágenes
                            de tipo "header" o "header_mobile"
                        */
                        $query->imagesOnly()->whereIn('type_id', [
                            $headerFiletype->id,
                            $headerMobileFiletype->id
                        ]);
                    })
                    ->with(['mediafiles' => function ($query) use ($headerFiletype, $headerMobileFiletype){
                        /* Sólo cargar las imágenes de tipo "header" o "header_mobile" */
                        $query->imagesOnly()->whereIn('type_id', [
                            $headerFiletype->id,
                            $headerMobileFiletype->id
                        ]);
                    }])
                    ->with([$featurableMethod])/* Cargar la fuente de imagen */
                    ->get()
                    ->groupBy('site_id')
                    ->mapWithKeys(function ($pages, $siteId) use ($featurableMethod, $featurableDateFormat, $now, $featurable) {
                        /** @var Collection|Collection[] Lista de datos en fuente */
                        $siteFueatureSources = $pages->map(function ($page) use ($featurableMethod, $featurableDateFormat, $now, $featurable){
                            /* Obtener la fuente de imágenes */
                            $feature = $page->{$featurableMethod};

                            /* Establecer la fecha de inicio, al principio del día */
                            $startDate = Carbon::createFromFormat(
                                $featurableDateFormat,
                                $feature->featurable_start_date
                            )->startOfDay();

                            /* Establecer la fecha de fin, al fin del día */

                            $endDate = Carbon::createFromFormat(
                                $featurableDateFormat,
                                $feature->featurable_end_date
                            )->endOfday();

                            /*
                                Obtener los estados de las páginas
                                por traducción
                             */
                            $pageLocaleStatuses = $page->translations->mapWithKeys(function ($translation) {
                                return [
                                    "{$translation->locale}" => $translation->status,
                                ];
                            });

                            return collect([
                                'featurable_type' => $featurable,
                                'featurable_id' => $feature->id,
                                'start_date' => $startDate,
                                'end_date' => $endDate,
                                'permanent' => $feature->featurable_permanent,
                                'mediafiles' => $page->mediafiles,
                                'page_translation_status' => $pageLocaleStatuses,
                            ]);
                        })->filter(function ($value) use ($now){
                            /* Es vigente si la fecha está entre las fechas establecidas o si está puesto como permanente */
                            return
                                $now->isBetween(
                                    $value->get('start_date'),
                                    $value->get('end_date'),
                                    true
                                )
                                ||
                                $value->get('permanent');
                        });

                        /* Retornar definiciones por ID de sitio */
                        return [
                            "{$siteId}" => $siteFueatureSources,
                        ];
                    });

                /* Añadir las fuentes encontradas a la lista */
                $currentFeatureSourcesPerSiteId->each(function ($sources, $siteId)  use (&$featureSourcesPerSiteId) {
                    /* Obtener y mezclar */
                    $addedFeatureSources = $featureSourcesPerSiteId
                        ->get($siteId)
                        ->merge($sources);

                    /* Actualizar espacio con fuentes añadidas */
                    $featureSourcesPerSiteId
                        ->put($siteId, $addedFeatureSources);
                });
            });

            /*
                Para este momento, $featureSourcesPerSiteId ya tiene todas las
                fuentes por sitio. Es momento de hacer los trabajos de inyección
            */

            /**
             * Espacio para recabar las acciones a realizar en cada galería
             *
             * Collection([
             *    Collection([
             *        'delete' => (Collection|int[]),
             *        'updateOrCreate' => Collection([
             *            Collection([
             *                'id' => (int), menor a 1 = crear, 1 o más = actualizar
             *                'mediafile_id' => (int),
             *                'gallery_id' => (int),
             *                ...
             *            ]),
             *            ...
             *        ])
             *    ]),
             *    ...
             * ])
             *
             * @var Collection|Collection[]
             */
            $galleriesDataActions = collect();

            /** @var Collection|Item[] Items de las galerías solicitadas */
            $allItemsAlreadyIn = $this->itemRepository->getByGallery($allFeatures->pluck('gallery_id'));

            /** @var Collection|Gallery[] Galerías dónde inyectar */
            $galleriesPerSiteId = $allFeatures->pluck('gallery')->groupBy('page.site_id');

            /* Ciclar entre los sitios... */
            $galleriesPerSiteId->each(function ($galleries, $siteId) use ($allItemsAlreadyIn, $featureSourcesPerSiteId, $allFeatures, $now, &$galleriesDataActions, $headerFiletype, $headerMobileFiletype) {
                /* ... y las galerías */
                $galleries->each(function ($gallery) use ($allItemsAlreadyIn, $featureSourcesPerSiteId, $allFeatures, $now, &$galleriesDataActions, $headerFiletype, $headerMobileFiletype, $siteId) {
                    /* Obtener los que ya están dentro */
                    $itemsAlreadyIn = $allItemsAlreadyIn->where('gallery_id', $gallery->id);

                    /* Obtener las fuentes para esta galería */
                    $featuresForGallery = $allFeatures->where('gallery_id', $gallery->id);

                    /* Obtener y ordenar las definiciones */
                    $sourcesToUse = $featureSourcesPerSiteId->get($siteId)->whereIn(
                        'featurable_type',
                        $featuresForGallery
                            ->pluck('featurable_type')
                            ->all()
                    )->sortBy(function ($source) {
                        /* Ordenar ascendentemente por valor entero de fecha UNIX */
                        return $source->get('start_date')->getTimestamp();
                    });

                    /*
                      Extraer los mediafiles. El orden establecido anteriormente
                      es respetado.
                    */

                    /**
                     * Colecciones con fuente, ID y mediafile
                     * @var Collection|Collection[]
                     */
                    $sourcedMediafiles = $sourcesToUse->map(function($source) use ($gallery, $headerFiletype, $headerMobileFiletype) {
                        /* Espacio para poner las colecciones con datos */
                        $mediafileCollection = collect();

                        /* Filtrar los mediafiles a usar de acuerdo con la conducta móvil */
                        $mediafilesFromSource = ($gallery->is_mobile)
                            ? $source->get('mediafiles')->where('type_id', $headerMobileFiletype->id)
                            : $source->get('mediafiles')->where('type_id', $headerFiletype->id);

                        $mediafilesFromSource
                            ->each(function($mediafile) use ($source, &$mediafileCollection) {
                                $mediafileCollection->push(
                                    collect([
                                        'featurable_type' => $source->get('featurable_type'),
                                        'featurable_id' => $source->get('featurable_id'),
                                        'mediafile' => $mediafile,
                                    ])
                                );
                            });

                        /* Retornar la colección de colecciones */
                        return $mediafileCollection;
                    })->flatten(1);
                    /* Sólo aplanar un nivel */

                    /*
                        Discernir cuáles de los que ya están deben ser eliminados:
                        - Tienen tipo de fuente (no es null)
                        - No están en la colección de mediafiles nuevos
                    */

                    /**
                     * Ítems que deben de ser removidos.
                     * @var Collection|Item[]
                     */
                    $itemsToRemove = $itemsAlreadyIn->filter(function ($item) use ($sourcedMediafiles) {
                        return (
                            $item->featurable_type != null
                            &&
                            ($sourcedMediafiles->where('mediafile.id', $item->mediafile_id)->count() < 1)
                        );
                    });

                    /* Remover de la lista de incluídos los que deben ser removidos */
                    $itemsToUpdate = $itemsAlreadyIn->filter(function ($item) use ($itemsToRemove) {
                        return $itemsToRemove->where('mediafile_id', $item->mediafile_id)->count() < 1;
                    });

                    /* Discernir cuáles son los mediafiles que hay que incluir */
                    $mediafilesToInsert = $sourcedMediafiles->filter(function($mediafile) use ($itemsToUpdate, $itemsToRemove){
                        /* Mantener si no está insertado y no hay que remover */
                        return (
                            (
                                $itemsToUpdate
                                    ->where('mediafile_id', $mediafile->get('mediafile')->id)
                                    ->count() < 1
                            )
                            &&
                            (
                                $itemsToRemove
                                    ->where('mediafile_id', $mediafile->get('mediafile')->id)
                                    ->count() < 1
                            )
                        );
                    });

                    /*
                        Para este momento, $mediafilesToInsert tiene los mediafiles que
                        deben de insertarse a esta galería. Buscar el orden manual
                        previamente establecido y respetarlo.
                    */

                    /**
                     * Lista de datos fijos a actualizar en la galería (cambiar orden)
                     * @var Collection|Collection[]
                     */
                    $fixedDataToUpdate = $itemsToUpdate->filter(function ($item) {
                        return ($item->is_priority_fixed == true);
                    })->sortBy('priority')->map(function ($item) use ($now) {
                        return collect([
                            'id' => $item->id,
                            'mediafile_id' => $item->mediafile_id,
                            'gallery_id' => $item->gallery_id,
                            'updated_at' => $now->toDateTimeString(),
                        ]);
                    });

                    /**
                     * Lista de datos dinámicos a actualizar en la galería (cambiar orden)
                     * @var Collection|Collection[]
                     */
                    $dynamicDataToUpdate = $itemsToUpdate->filter(function ($item) {
                        return ($item->is_priority_fixed != true);
                    })->map(function ($item) use ($now) {
                        return collect([
                            'id' => $item->id,
                            'mediafile_id' => $item->mediafile_id,
                            'gallery_id' => $item->gallery_id,
                            'updated_at' => $now->toDateTimeString(),
                        ]);
                    });

                    /**
                     * Lista de datos a insertar a la galería
                     * @var Collection|Collection[]
                     */
                    $dynamicDataToInsert = $mediafilesToInsert->map(function ($mediafile) use ($now, $gallery) {
                        return collect([
                            'id' => 0,
                            'mediafile_id' => $mediafile->get('mediafile')->id,
                            'gallery_id' => $gallery->id,
                            'featurable_type' => $mediafile->get('featurable_type'),
                            'featurable_id' => $mediafile->get('featurable_id'),
                            'created_at' => $now->toDateTimeString(),
                        ]);
                    });

                    /**
                     * Lista de datos a cambiar a la galería, en el orden correcto.
                     *
                     * Si la colección de datos tiene ID de 0 o menos, la acción
                     * será inserción.
                     *
                     * @var Collection|Collection[]
                     */
                    $dynamicDataToNewOrder = $dynamicDataToInsert->merge($dynamicDataToUpdate)->sortBy(function ($data) use ($sourcedMediafiles) {
                        /* Regresar el índice desde la lista ordenada */
                        return $sourcedMediafiles->search(function ($mediafile) use ($data) {
                            /* Encontrar modelo con el mismo ID */
                            return $mediafile->get('mediafile')->id == $data->get('mediafile_id');
                        });
                    });

                    /*
                        la información está lista para hacer los cambios necesarios
                        a esta galería:

                        $itemsToRemove (eliminar ítems)
                        $dynamicDataToNewOrder (ítems dinámicos a crear/actualizar,
                                                ya ordenados)
                        $fixedDataToUpdate (ítems fijos a actualizar, ya ordenados)

                        Necesario unir los elementos fijos al final de los dinámicos
                        y establecer nuevo orden en los elementos.
                    */

                    /**
                     * Lista de datos listos para la galería, en el orden correcto.
                     *
                     * Si la colección de datos no tiene ID, entonces hay
                     * que insertarlo.
                     *
                     * @var Collection|Collection[]
                     */
                    $allDataToNewOrder = $dynamicDataToNewOrder->merge($fixedDataToUpdate)->map(function ($data, $index) {
                        /* Establecer prioridad */
                        $data->put('priority', $index);

                        /* Retornar nuevos datos */
                        return $data;
                    });

                    /** @var Collection|Collection[] Acciones para esta con datos */
                    $galleryDataActions = collect([
                        /* Colección de IDs */
                        'delete' => $itemsToRemove->pluck('id'),
                        'updateOrCreate' => $allDataToNewOrder,
                    ]);

                    /* Establecer as acciones para esta galería */
                    $galleriesDataActions->put($gallery->id, $galleryDataActions);
                });
            });

            /*
                Para este momento, $galleriesDataActions ya tiene todas las acciones
                de eliminación, actualización e inserción por galería existente.

                Primero, obtener todos los ítems a eliminar, los datos a actualizar
                y los datos a crear.
             */

            /** @var Collection|int[] IDs de todos los ítems a eliminar */
            $idsAllItemsToDelete = $galleriesDataActions->pluck('delete')->flatten()->unique();

            /**
             * Datos de todos los ítems a actualizar o crear
             *
             * @var Collection|Collection[]
             */
            $allItemsToUpdateOrCreate = $galleriesDataActions->pluck('updateOrCreate')->flatten(1);

            /**
             * Datos de todos los ítems a sólo actualizar
             *
             * @var Collection|Collection[]
             */
            $allItemDataToUpdate = $allItemsToUpdateOrCreate->filter(function($itemData) {
                /* Los ítems con ID mayor a 0 deben ser actualizados */
                return $itemData->get('id') > 0;
            });

            /**
             * Datos de todos los ítems a sólo crear
             *
             * @var Collection|int[]
             */
            $allItemDataToInsert = $allItemsToUpdateOrCreate->filter(function($itemData) {
                /* Los ítems con ID menor a 1 deben ser creados */
                return $itemData->get('id') < 1;
            });

            /* Todos los datos están separados. Ejecutar... */
            $this->info("Información compilada. Ejecutando...");

            /* Eliminar los ítems */
            if ($idsAllItemsToDelete->isNotEmpty()) {
                $this->line("Eliminando ítems existentes...");

                $this->itemRepository
                    ->whereIn('id', $idsAllItemsToDelete->all())
                    ->delete();

                $this->info("Ítems eliminados.");
            }

            /* Actualizar los ítems existentes */
            if ($allItemDataToUpdate->isNotEmpty()) {
                $this->line("Nulificar ordenamientos actuales...");

                /* Nulificar todos los ordenamientos */
                $this->itemRepository
                    ->whereIn(
                        'gallery_id',
                        $allItemDataToUpdate->pluck('gallery_id')->unique()->all()
                    )->update([
                        "priority" => null,
                    ]);

                $this->info("Ordenamientos nulificados.");

                $this->line("Actualizando orden de elementos establecidos...");

                /*
                   Actualizar con nuevos órdenes. Necesario construir consulta cruda
                   para evitar ciclo con múltiples llamadas
                */

               /** @var int Número de registros actualizados */
                $updatedItems = $this->itemRepository->query()
                    ->setBindings(
                        array_merge(
                            /* NOTA: Orden es imperativo */
                            $allItemDataToUpdate->mapWithKeys(function ($itemData) {
                                $itemId = $itemData->get('id');
                                $itemPriority = $itemData->get('priority');

                                /* Retornar nuevo biding con valor de prioridad */
                                return [
                                    "priorityId{$itemId}" => (int)$itemPriority,
                                ];
                            })->all(),
                            $allItemDataToUpdate->mapWithKeys(function ($itemData) {
                                $itemId = $itemData->get('id');

                                /* Retornar nuevo biding con valor de tiempo */
                                return [
                                    "updatedAtId{$itemId}" => $itemData->get('updated_at'),
                                ];
                                /* Necesario establecer `updated_at` porque, si no,
                                 el sistema lo implementa usando el binding
                                 por posiciones */
                            })->all()
                        )
                    )
                    ->update([
                        'priority' => DB::raw(
                            implode([
                                'CASE `id`',
                                $allItemDataToUpdate->map(function ($itemData) {
                                    $itemId = $itemData->get('id');

                                    /* Retornar SQL de caso */
                                    return "WHEN {$itemId} THEN :priorityId{$itemId}";
                                })->implode(' '),
                                'ELSE `priority` END'
                            ], ' ')
                        ),
                        'updated_at' => DB::raw(
                            implode([
                                'CASE `id`',
                                $allItemDataToUpdate->map(function ($itemData) {
                                    $itemId = $itemData->get('id');

                                    /* Retornar SQL de caso */
                                    return "WHEN {$itemId} THEN :updatedAtId{$itemId}";
                                })->implode(' '),
                                'ELSE `updated_at` END'
                            ], ' ')
                        ),
                    ]);

                $this->info("{$updatedItems} elementos actualizados.");
            }

            /* Insertar los ítems nuevos */
            if ($allItemDataToInsert->isNotEmpty()) {

                $this->line("Creando nuevos elementos...");

                /*
                   Creación de nuevos elementos. Necesario construir consulta cruda
                   para evitar ciclo con múltiples llamadas
                */

                /** @var int Número de registros actualizados */
                $createdItems = $this->itemRepository->insert(
                    $allItemDataToInsert->map(function($itemData) {
                        return [
                            'gallery_id' => $itemData->get('gallery_id'),
                            'mediafile_id' => $itemData->get('mediafile_id'),
                            'priority' => $itemData->get('priority'),
                            'featurable_type' => $itemData->get('featurable_type'),
                            'featurable_id' => $itemData->get('featurable_id'),
                            'is_priority_fixed' => false,
                            /* Añadidos dinámicamente */
                            'created_at' => $itemData->get('created_at'),
                            /* insert() no lo establece de
                            forma automática */
                        ];
                    })->all()
                );

                $this->info("{$createdItems} elementos creados.");
            }

            /* Aplicar cambios */
            DB::commit();
        } catch (Exception $e) {
            /* Desbacer cambios a la base de datos */
            DB::rollback();

            /* Mensaje para mostrar */
            $errorMessage = $e->getMessage();
            $errorFile = $e->getFile();
            $errorLine = $e->getLine();

            /* Informar al usuario del error */
            $this->error("Error al intentar actualizar las galerías: {$errorMessage} en el archivo {$errorFile}, línea {$errorLine}");
        }

        /* Informar a usuario de finalización */
        $this->info('Actualización de galerías terminada.');
    }
}
