<?php

namespace AvengersMG\MGCms2019\App\Console\Commands;


use AvengersMG\MGCms2019\App\Cms\Pages\Repositories\PageInterface;
use AvengersMG\MGCms2019\App\Cms\Components\Specials\Special;
use AvengersMG\MGCms2019\App\Cms\Components\Specials\SpecialInterface;
use AvengersMG\MGCms2019\App\Cms\Components\Specials\SpecialNotification;
use AvengersMG\MGCms2019\App\Cms\Sites\Repositories\SiteInterface;
use AvengersMG\MGCms2019\App\Cms\Sites\Site;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class SpecialsPushNotfication extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'mgcms2019:send-specials-push-notifications';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Permite el envío de notificaciones push con horario personalizado tomando hora del servidor ejecutando un cron job';

    protected $special;
    protected $siteRepository;
    protected $pageRepository;
    protected $specialRepository;
    /**
     * Create a new command instance.
     *
     * @return void
     */

     /**
     * Espacio para el repositorio de specials
     *
     * @var SpecialFeatureInterface
     */
    protected $specialFeatureRepository;

    public function __construct( SpecialInterface $specialInfterface,
                                 PageInterface $pageInterface,
                                 Special $special,
                                 SiteInterface $siteInterface)
    {
        parent::__construct();

        $this->special = $special;
        $this->pageRepository = $pageInterface;
        $this->specialRepository = $specialInfterface;
        $this->siteRepository = $siteInterface;
    }

    /**
     * Execute the console command.
     *
     * @return mixed
     */
    public function handle()
    {
        /* Obtener la fecha de hoy, para usos múltiples */
        $now = Carbon::now('America/Mexico_City');

        try{
            $this->line('Enviando notificaciones...');

            $sites = $this->siteRepository->all();

            $allSpecialActivesBySite = $this->specialRepository->getActivesBySite($sites)->get();

            /* Variable para mantener contexto */
            $self = $this;

            /** @var Collection|SpecialNotification[] Notificaciones en cola */
            $specialNotificationsPending = $allSpecialActivesBySite->pluck('translations')->flatten(1)->pluck('notifications')->flatten(1)->filter(function($value, $index) use($now, $self) {
                if($now->greaterThanOrEqualTo($value->notification_time) && is_null($value->sent_at)){
                    $self->line('Hay fechas coincidentes');
                    return true;
                } else {
                    $self->line("No hay fechas coincidentes: {$value->notification_time}");
                    return false;
                }
            });

            /**
             * Obtener IDs de las traducciones en las notificaciones
             * @var Collection|int[]
             */
            $specialNotificationsPendingIds = $specialNotificationsPending->pluck('special_translation_id')->values();

            /** @var Collection|Special[] Obtener los especiales para enviar */
            $specialsToSend = $allSpecialActivesBySite->filter(function($special) use ($specialNotificationsPendingIds) {
                /**
                 *  Si el especial contiene traducciones pendientes de enviar,
                 *  incluir en el envío
                 * @var boolean
                 */
                return $special->translations->whereIn('id', $specialNotificationsPendingIds->all())->isNotEmpty();
            });

            /* Hacer el envío por cada especial nuevo */
            $specialsToSend->each(function($special) use ($self){
                /* Enviar nueva notificación */
                $self->specialRepository->createSpecialEvent($special);
            });

            /* Solo actualizar si sí hay especiales a enviar */
            if ($specialsToSend->isNotEmpty()) {
                /* Notificaciones enviadas. Actualizar registros */
                SpecialNotification::query()
                    ->setBindings(
                        $specialNotificationsPendingIds->mapWithKeys(function ($specialTransId) use ($now) {
                            /* Retornar nuevo biding con valor de prioridad */
                            return [
                                "id{$specialTransId}" => $now->toDateTimeString(),
                            ];
                        })->all()
                    )
                    ->update([
                        'sent_at' => DB::raw(
                            implode([
                                'CASE `special_translation_id`',
                                $specialNotificationsPendingIds->map(function ($specialTransId) {
                                    /* Retornar SQL de caso */
                                    return "WHEN {$specialTransId} THEN :id{$specialTransId}";
                                })->implode(' '),
                                'ELSE `sent_at` END'
                            ], ' ')
                        )
                    ]);

                $this->info('Registros actualizados.');
            }

            $this->line('Proceso terminado...');

        } catch (Exception $e) {

             /* Mensaje para mostrar */
             $errorMessage = $e->getMessage();
             $errorFile = $e->getFile();
             $errorLine = $e->getLine();
 
             /* Informar al usuario del error */
             $this->error("Error al intentar enviar notificaciones: {$errorMessage} en el archivo {$errorFile}, línea {$errorLine}");
        }

        $this->info('Notificaciones enviadas.');

    }
}