<?php

namespace AvengersMG\MGCms2019\App\Console\Commands;

use \Exception;
use AvengersMG\MGCms2019\App\Cms\Components\Specials\SpecialInterface;
use AvengersMG\MGCms2019\App\Cms\Pages\PageTranslation;
use AvengersMG\MGCms2019\App\Cms\Pages\Repositories\PageInterface;
use AvengersMG\MGCms2019\App\Cms\Sites\Repositories\SiteInterface;
use Carbon\Carbon;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class HideExpiredSpecials extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'mgcms2019:hide-expired-specials';

    /**
     * Espacio para el repositorio de especiales (ofertas)
     *
     * @var SpecialInterface
     */
     protected $specialRepository;

     /**
      * Espacio para el repositorio de páginas
      *
      * @var PageInterface
      */
     protected $pageRepository;

    /**
     * Espacio para el repositorio de sitios
     *
     * @var SiteInterface
     */
    protected $siteRepository;


    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Remover de la vista las páginas que contengan especiales caducos (no permanentes y que la ventana de reserva ya pasó) para uso del paquete AvengersMG\\MGCms2019.';

    /**
     * Create a new command instance.
     *
     * @param SpecialInterface        $specialInterface         Repositorio de especiales (ofertas)
     * @param PageInterface           $pageInterface            Repositorio de páginas
     * @param SiteInterface           $siteInterface            Repositorio de sitios
     * @return void
     */
    public function __construct(
        SpecialInterface $specialInterface,
        PageInterface $pageInterface,
        SiteInterface $siteInterface
    )
    {
        parent::__construct();

        /* Provisionar comando */
        $this->specialRepository = $specialInterface;
        $this->pageRepository = $pageInterface;
        $this->siteRepository = $siteInterface;
    }

    /**
     * Execute the console command.
     *
     * @return mixed
     */
    public function handle()
    {
        /* Obtener la fecha de hoy, para usos múltiples */
        $now = Carbon::now();

        /* Iniciar transacción */
        DB::beginTransaction();

        try {
            $this->line('Buscando páginas con especiales no permanentes...');

            /**
             * Todas las páginas que tengan especiales (ofertas) no permanentes
             * y no estén escondidas
             *
             * @var Collection|Page[]
             */
            $specialPages = $this->pageRepository->whereHas('special', function ($query) {
                /* Solo cuando el especial no sea permanente (null o false) */
                $query
                    ->where('permanent', '!=', true)
                    ->orWhereNull('permanent');
            })->whereHas('translations', function ($query) {
                /* Solo cuando el estado no sea "hidden" */
                $query->where('status', '!=', 'hidden');
            })->with('special')->get();

            $this->line('Filtrando resultados por fecha actual...');

            /**
             * Todas las páginas que tengan especiales (ofertas) no permanentes,
             * no estén escondidas y la fecha actual sea mayor que la fechas
             * de fin
             *
             * @var Collection|Page[]
             */
            $expiredSpecialPages = $specialPages->filter(function ($page) use ($now) {
                /* Obtener la fecha de fin (fin de la ventana de reservación) */
                $endDate = Carbon::createFromFormat(
                    $page->special->getFeaturableDateFormat(),
                    $page->special->end_date
                )->endOfday();

                /* Si la fecha actual es mayor que la fecha de fin, retornar verdadero */
                return $now->greaterThan($endDate);
            });

            /* Si hay ofertas caducas, cambiar estado a "hidden" */
            if ($expiredSpecialPages->isNotEmpty()) {

                $this->line('Ofertas caducas encontradas. Actualizando estado...');

                /* Actualizar estados */
                $updatedTranslations = PageTranslation::whereIn('page_id', $expiredSpecialPages->pluck('id')->all())
                    ->update([
                        'status' => 'hidden',
                    ]);

                $this->info("{$updatedTranslations} traducciones de ofertas actualizadas.");
            } else {
                $this->line('No hubo ofertas caducas.');
            }

            /* Aplicar cambios */
            DB::commit();
        } catch (Exception $e) {
            /* Desbacer cambios a la base de datos */
            DB::rollback();

            /* Mensaje para mostrar */
            $errorMessage = $e->getMessage();
            $errorFile = $e->getFile();
            $errorLine = $e->getLine();

            /* Informar al usuario del error */
            $this->error("Error al intentar actualizar las traducciones de ofertas caducas: {$errorMessage} en el archivo {$errorFile}, línea {$errorLine}");
        }

        /* Informar a usuario de finalización */
        $this->info('Actualización de ofertas terminada.');
    }
}
