<?php

namespace AvengersMG\MGCms2019\App\Console\Commands;

use \Exception;
use AvengersMG\MGCms2019\App\Cms\Availabilities\AvailabilitiesLogEntry;
use Carbon\Carbon;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class DeleteStaleAvailabilityLogEntries extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'mgcms2019:delete-stale-availability-log-entries';


    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Limpiar la tabla de registro de últimos cambios a la tabla de disponibilidades. Removerá las entradas que tengan más de 30 días, con excepción de las últimas 10, para uso del paquete AvengersMG\\MGCms2019.';

    /**
     * Execute the console command.
     *
     * @return mixed
     */
    public function handle()
    {
        /* Obtener la fecha de hoy, para usos múltiples */
        $now = Carbon::now();

        /*
            Cantidad mínima de registros. Si la cantidad de registros es mayor,
            el sistema hará limpieza en los que sobran
         */
        $minimumCount = 10;

        /* Iniciar transacción */
        DB::beginTransaction();

        try {
            $this->line('Contando entradas actuales...');

            $count = AvailabilitiesLogEntry::count();

            /* Solo efectuar borrado si la cuenta de entradas es mayor a 10 */
            if ($count >= $minimumCount) {
                $this->info("La tabla contiene {$count} entradas. Inicia la limpieza...");

                /**
                 * Registros caducos, saltando los 10 más recientes
                 *
                 * @var Collection|integer[]
                 */
                $staleEntriesIds = AvailabilitiesLogEntry::latest()
                    ->take($count) /* Obtener la cuenta */
                    ->skip($minimumCount) /* De la cuenta anterior, saltar */
                    ->get()
                    ->filter(function ($entry) use ($now) {
                        /* Si el registro fue mayor a 30 días, eliminar */
                        return $entry->created_at->diffInDays($now, false) > 30;
                    })
                    ->map(function ($entry) {
                        /* Retornar solo el ID */
                        return $entry->id;
                    });

                if ($staleEntriesIds->isNotEmpty()) {
                    $this->info("Intentando eliminar {$staleEntriesIds->count()} entradas...");

                    /* Enviar IDs para destrucción */
                    AvailabilitiesLogEntry::destroy($staleEntriesIds);

                    $this->info("Entradas eliminadas.");
                } else {
                    $this->info("No hay entradas para eliminar.");
                }
            } else {
                $this->info("La tabla contiene {$minimumCount} o menos entradas. Operación abortada.");
            }

            /* Aplicar cambios */
            DB::commit();
        } catch (Exception $e) {
            /* Desbacer cambios a la base de datos */
            DB::rollback();

            /* Mensaje para mostrar */
            $errorMessage = $e->getMessage();
            $errorFile = $e->getFile();
            $errorLine = $e->getLine();

            /* Informar al usuario del error */
            $this->error("Error al intentar eliminar entradas caducas de actualización de disponibilidad: {$errorMessage} en el archivo {$errorFile}, línea {$errorLine}");
        }

        /* Informar a usuario de finalización */
        $this->info('Remoción de entradas terminada.');
    }
}
