<?php

namespace AvengersMG\MGCms2019\App\Cms\Sites;

use Astrotomic\Translatable\Translatable;
use AvengersMG\MGCms2019\App\Cms\Users\User;
use AvengersMG\MGCms2019\App\Cms\Pages\Page;
use AvengersMG\MGCms2019\App\Cms\Socialmedias\Socialmedia;
use AvengersMG\MGCms2019\App\Cms\Mediafiles\Mediafile;
use Illuminate\Database\Eloquent\Model;

class Site extends Model
{
    use Translatable;

    public $appends = ['base_url'];

    public $translatedAttributes = ['meta_title', 'meta_description','phone','schema'];

    protected $fillable = ['domain', 'name', 'logo'];

    public function users()
    {
        return $this->belongsToMany(User::class);
    }

    public function socialmedias()
    {
        return $this->belongsToMany(Socialmedia::class)->withPivot('url');
    }

    public function pages()
    {
        return $this->hasMany(Page::class);
    }

    /**
     * Método público para recuperar la página que funge como lista de precios
     * de habitaciones
     *
     * @return Page|null    La página o null si no existe
     */
    public function getRoomPricesPage()
    {
        return $this->pages()->where('is_roomprices_list', true)->first();
    }

    public function mediafiles()
    {
        return $this->morphMany(Mediafile::class, 'mediafilable');
    }


    public function setNameAttribute($value)
    {
        $this->attributes['name'] = mb_strtolower($value);
    }

    /**
     * Método público para recuperar una URL base absoluta del sitio web
     * @return string La URL base absoluta
     */
    public function getBaseUrlAttribute()
    {
        $raw_schema = "http";
        $schema = "{$raw_schema}://";
        $secure_schema = "{$raw_schema}s://";
        $raw_domain = "";

        /** @var boolean Evaluación si el valor empieza con HTTP */
        $starts_insecure = starts_with($this->domain, $schema);

        /** @var boolean Evaluación si el valor empieza con HTTPS */
        $starts_secure = starts_with($this->domain, $secure_schema);

        /** @var boolean Evaluación si la petición principal es segura */
        $request_is_secure = request()->secure();

        if (
            /* Si el dominio ya tiene HTTPS... */
            ($starts_secure)
            ||
            /* o si el dominio ya tiene HTTP y seguridad NO es exigida... */
            ($starts_insecure && !$request_is_secure)
        ) {
            /* No hace falta modificar. Retornar */
            return $this->domain;
        } else if ($starts_insecure && $request_is_secure) {
            /* Si no tiene HTTPS y exige seguridad, construir */

            /* Obtener el dominio crudo */
            $raw_domain = substr($this->domain, strlen($schema));

            /* Dirección con el schema correcto */
            return "{$secure_schema}{$raw_domain}";
        } else if (!$starts_insecure && !$starts_secure && $request_is_secure) {
            /* Si no tiene HTTPS ni HTTP y exige seguridad, construir */

            /* Dirección con el schema correcto */
            return "{$secure_schema}{$this->domain}";
        }

        /* Dirección con el schema predeterminado */
        return "{$schema}{$this->domain}";
    }

    /**
     * Método estático público para retornar el objeto pertinente al dominio actual
     * @return Site El sitio actual
     */
    public static function current()
    {
        /* Obtener dominio de la petición */
        $currentHost = request()->getHttpHost();

        /* Buscar sitio de acuerdo con el dominio */
        return (static::count() > 1)? static::whereIn('domain', [
            $currentHost,
            "www.{$currentHost}"
        ])->first() : static::first();
    }
}
