<?php

namespace AvengersMG\MGCms2019\App\Cms\Pages\Repositories;

use \Exception;
use AvengersMG\MGCms2019\App\Cms\Accommodations\Amenity;
use AvengersMG\MGCms2019\App\Cms\Accommodations\Category;
use AvengersMG\MGCms2019\App\Cms\Components\Galleries\GalleryFeature;
use AvengersMG\MGCms2019\App\Cms\BaseRepository\BaseRepository;
use AvengersMG\MGCms2019\App\Cms\Mediafiles\Mediafile;
use AvengersMG\MGCms2019\App\Cms\Pages\Page;
use AvengersMG\MGCms2019\App\Cms\Pages\PageTranslation;
use AvengersMG\MGCms2019\App\Cms\Pages\Repositories\PageInterface;
use AvengersMG\MGCms2019\App\Cms\Sites\Site;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Illuminate\Database\Eloquent\Builder;

class PageRepository extends BaseRepository implements PageInterface
{
    public function __construct()
    {
        parent::__construct(Page::class);
    }

    /**
     * Método público para recuperar las IDs de todas las páginas del sitio especificado
     *
     * @param  Site   $site       Variable de sitio
     * @return Collection|int[]   Colección de IDs
     */
    public function getPagesIdsBySite(Site $site)
    {
        return call_user_func("{$this->model}::bySite", $site)->pluck('id');
    }

    /**
     * Método público para construir un colección con la jerarquía establecida
     * en la base de datos
     *
     * Incluirá las traducciones de propiedades (sin las traduciones de las relaciones)
     *
     * @param  Site|null                    $site Variable de sitio
     * @return Kalnoy\Nestedset\Collection        Jerarquía completa de páginas
     */
    public function toTree($site = null)
    {
        /* Sólo escoger sitio si está definido */
        $pages = (!is_null($site)) ? $site->pages() : call_user_func("{$this->model}::bySite", Site::current())->all();

        /* Retornar con traducciones */
        return $pages->with('translations')->defaultOrder()->get()->toTree();
    }

    /**
     * Método público para construir un colección con la jerarquía establecida
     * en la base de datos
     *
     * @param  Site|null                    $site Variable de sitio
     * @return Kalnoy\Nestedset\Collection        Jerarquía completa de páginas
     */
    public function toTrees($site = null)
    {
        return $this->listWithPropertyTranslations($site)->toTree();
    }

    /**
     * Método público para construir una colección con todas las páginas, pero que también incluirá las traducciones de propiedades (sin las traduciones de las relaciones)
     *
     * @param  Site|null                    $site Variable de sitio
     * @return Kalnoy\Nestedset\Collection        Colección de páginas
     */
    public function listWithPropertyTranslations($site = null)
    {
        return call_user_func("{$this->model}::bySite", ((is_null($site)) ? Site::current() : $site))->with(['translations'])->defaultOrder()->get();
    }

    /**
     * Método público para construir una colección con todas las páginas, pero que también incluirá las traducciones de propiedades (CON las traduciones de las relaciones)
     *
     * @param  Site|null                    $site Variable de sitio
     * @return Kalnoy\Nestedset\Collection        Colección de páginas
     */
    public function listWithRelationsTranslations($site = null)
    {
        return call_user_func("{$this->model}::bySite", ((is_null($site)) ? Site::current() : $site))->with(['translations', 'cards.translations'])->defaultOrder()->get();
    }

    public function mediaFile()
    {
        return call_user_func("{$this->model}::mediafiles");
    }

    public function getRooms()
    {
        return call_user_func("{$this->model}::room");
    }

    /**
     * Método público para reconstruir la jerarquía del modelo en el sitio actual
     *
     * No es necesario enviar el sitio porque la jerarquía está basada en los IDs de habitaciones
     *
     * @param  array[]                        $data Arreglo de arreglos
     *                                              con etiquetas (['id' => <id>,
     *                                              'children' => [...]])
     * @return Kalnoy\Nestedset\Collection          Jerarquía completa de páginas
     */
    public function rebuildTree($data)
    {
        /* Obtener sitio del primer elemento (es el mismo para todos). Debe ser redirigido al modelo */
        $site = $this->find($data[0]['id'])->site;

        /* Iniciar transacción */
        DB::beginTransaction();

        try {
            /* Reconstruir jerarquía */
            call_user_func("{$this->model}::rebuildTree", $data);

            /* Actualizar permalinks */
            $this->rewritePermaLinks($site);

            /* Cambios realizados correctamente */
            DB::commit();
        } catch (Exception $e) {
            /* Deshacer cambios */
            DB::rollback();
        }

        return $this->toTrees($site);
    }

    /**
     * Método para actualizar los permalinks de un sitio completo
     *
     * Implementación pública y transaccional de $this->rewritePermaLinks()
     *
     * @param  Site|null   $site Variable de sitio
     * @return int               Cantidad de registros afectados
     */
    public function updatePermaLinks($site = null)
    {
        /** @var int Cantidad de registros afectados */
        $retorno = 0;

        /* Iniciar transacción */
        DB::beginTransaction();

        try {
            $retorno = $this->rewritePermaLinks($site);

            /* Cambios realizados correctamente */
            DB::commit();
        } catch (Exception $e) {
            /* Deshacer cambios */
            DB::rollback();
        }

        return $retorno;
    }

    public function isHome($slug)
    {
        if ($slug == '/' || $slug == '/es') {
            return true;
        }

        return false;
    }


    /**
     * Método público para la creación de objetos Page
     *
     * @param  mixed[]     $params Arreglo de parámetros
     * @return Page|null           Objeto Page si la creación fue exitosa
     */
    public function create($params)
    {
        $retorno = null;

        app()->setLocale($params['locale']);

        $params['seo'] = (empty($params['meta_title']) || empty($params['meta_description']))
            ? false
            : true;

        $params['perman_link'] = ($params['locale'] == 'es')
            ? ('/es' . $params['perman_link'])
            : $params['perman_link'];

        /* Iniciar transacción */
        DB::beginTransaction();

        try {
            /* $params ya debe incluir parent_id */
            $page = parent::create($params);

            /* Volver a cargar el modelo para que tenga los padres e hijos correctos incluidos */
            $page = $page->fresh();

            /* Al final de todo, actualizar los permalinks */
            $this->rewritePermaLinks($page->site);

            /* Cambios realizados correctamente */
            DB::commit();

            /* Actualizar valor de retorno */
            $retorno = $page;
        } catch (Exception $e) {
            /* Deshacer cambios */
            DB::rollback();

            /* Asegurar retorno vacío */
            $retorno = null;
        }

        return $retorno;
    }

    public function update($params, $model = '')
    {
        $updatingResult = false;

        /**
         * Espacio para información de modelo original (antes de cambio)
         *
         * @var array
         */
        $original = [];


        $data = $params->all();
        app()->setLocale($data['locale']);
        $lang = app()->getLocale();

        $data['perman_link'] = ($model->isRoot())
            ? $this->getSlug($lang, $data['slug'])
            : ($model->parent->perman_link . '/' . $data['slug']);

        if (!array_key_exists('navigation', $data)) {
            $data['navigation'] = false;
        }

        if (!array_key_exists('index', $data)) {
            $data['index'] = false;
        }

        /* null permite datos múltiples sin romper el índice único */
        if (!array_key_exists('is_roomprices_list', $data)) {
            $data['is_roomprices_list'] = null;
        }

        /* Iniciar transacción */
        DB::beginTransaction();

        try {
            /* Sitio establecido, para uso futuro */
            $site = $model->site;

            /* Obtener información de modelo original de traducción */
            $originalTranslation = ($model->hasTranslation($lang))
                ? $model->translate($lang)->getOriginal()
                : null;

            /* Actualizar lista de precios si es que debe establecerse */
            if (
                ((bool) $data['is_roomprices_list']) == true
                && ((bool) $model->is_roomprices_list) != true
            ) {
                /* Remover la página actual de reservaciones */
                $currentRoomPricesPage = $site->getRoomPricesPage();

                /* Si existe, desasociar antes de actualizar */
                if (!is_null($currentRoomPricesPage)) {
                    $currentRoomPricesPage->update([
                        'is_roomprices_list' => null,
                    ]);
                }
            }

            /* Actualizar modelo */
            $updatingResult = $model->update($data);

            /* Al final de todo, actualizar los permalinks */
            $this->rewritePermaLinks($site);

            /* Cambios realizdos correctamente */
            DB::commit();
        } catch (Exception $e) {
            /* Deshacer cambios */
            DB::rollback();

            /* Regresar al valor negativo */
            $updatingResult = false;
        }

        /* Si la actualización fue exitosa, ejecutar el comando de revisión */
        if ($updatingResult == true) {
            $this->triggerDynamicGalleriesUpdate(
                $model->translate($lang),
                $originalTranslation
            );
        }

        return $model;
    }

    public function getSlug($lang, $str)
    {
        if ($str != '/') {
            // SI NO ES EL HOME
            if ($lang == 'en') {
                return '/' . $str;
            } else {
                return '/es/' . $str;
            }
        } else {

            //SI ES HOME
            if ($lang == 'en') {
                return '/';
            } else {
                return '/es';
            }
        }
    }

    /**
     * Método público para la actualización de información de SEO
     *
     * @param  Request|UpdateSeoPageRequest $data  La información de la solicitud
     * @param  Model                        $model El modelo a actualizar
     * @return boolean                      Valor de éxito de acción
     */
    public function updateSeo($data, $model)
    {
        app()->setLocale($data->locale);

        /* Añadir valor a Request */
        $data->request->add([
            'seo' => (empty($data->meta_title) || empty($data->meta_description)) ? false : true
        ]);

        /* Retornar resultado */
        return $model->update($data->all());
    }

    /**
     * Método para la creación de permalinks
     *
     * @param  string[] $perman Arreglo de slugs
     * @return string           Permalink formado
     */
    public function generateSlug($perman)
    {
        /* Str::start() agregará "/" al principio si no existe */
        return Str::start(collect($perman)->map(function ($value) {
            return str_slug($value);
        })->implode('/'), '/');
    }

    /**
     * Método que retorna los diseños disponibles en el tema (resources/views/layouts)
     *
     * @return string[]      Arreglo de nombres de plantillas, con llaves de cadena
     */
    public function getLayouts()
    {
        $layouts = [];

        try {
            $files = scandir(base_path() . '/resources/views/layouts/');
            foreach ($files as $i => $layout) {
                if (strpos($layout, 'blade')) {
                    $string_parts = explode('.', $layout);
                    array_push($layouts, [$string_parts[0] => $string_parts[0]]);
                }
            }
        } catch (Exception $e) {
            return [];
        }

        return collect($layouts)->collapse();
    }

    /**
     * Método que retorna las plantillas disponibles en el tema (resources/views/)
     * @return string[]      Arreglo de nombres de plantillas, con llaves de cadena
     */
    public function getTemplates()
    {
        $templates = [];

        try {
            $files = scandir(base_path() . '/resources/views/templates/');

            foreach ($files as $i => $layout) {
                if (strpos($layout, 'blade')) {
                    $string_parts = explode('.', $layout);
                    array_push($templates, [$string_parts[0] => $string_parts[0]]);
                }
            }
        } catch (Exception $e) {
            return [];
        }

        return collect($templates)->collapse();
    }

    public function getCategories()
    {
        return Category::all();
    }

    public function getAmenities()
    {
        return Amenity::all();
    }

    /**
     * Método público para la eliminación de todos los modelos relacionados con la página enviada
     * @param  Page     $page Página a eliminar
     * @return boolean        Resultado de eliminación
     */
    public function deleteAll($page)
    {
        $retorno = false;

        /* Iniciar transacción */
        DB::beginTransaction();

        try {
            $page->deleteTranslations();

            if ($page->special) {
                $page->special->deleteTranslations();
                $page->special->delete();
            }

            if ($page->room) {
                $page->room->deleteTranslations();
                $page->room->delete();
            }

            $page->delete();

            /* Cambios realizados correctamente */
            DB::commit();

            /* Cambiar valor de retorno */
            $retorno = true;
        } catch (Exception $e) {
            /* Deshacer cambios */
            DB::rollback();
        }

        return $retorno;
    }


    /**
     * Front Methods
     */

    /**
     *  Método para obtener la navegación de un sitio
     *
     * @param  Site|null  $site  Variable de sitio
     * @return Collection|Page[] Lista de navegación
     */
    public function getNavigation($site = null)
    {
        $lang = app()->getLocale();

        return call_user_func("{$this->model}::bySite", ((is_null($site)) ? Site::current() : $site))
            ->with('translations')
            ->whereTranslation('navigation', 1, $lang)
            ->whereTranslation('status', 'published', $lang)
            ->defaultOrder()
            ->get()
            ->toTree();
    }

    /**
     * Método público para encontrar la página de acuerdo con el permalink
     *
     * @param  string     $value Permalink a buscar
     * @param  Site|null  $site  Variable de sitio
     * @return Page              Página encontrada
     */
    public function filterByPermalink($value, $site = null)
    {
        return call_user_func("{$this->model}::bySite", ((is_null($site)) ? Site::current() : $site))
            ->whereTranslation('perman_link', $value)
            ->first();
    }

    /**
     * Método público para la recuperación de páginas con especial activo
     *
     * @param  Site               $site   Variable de sitio
     * @return Builder                    Consulta filtrada
     */
    public function getActiveSpecialsBySite(Site $site)
    {
        return $this->with('special')
            ->where('site_id', $site->id)
            ->whereHas('special', function ($query) {
                $query
                    ->where(function ($specialQuery) {
                        $specialQuery
                            ->where('start_date', '<=', date('Y-m-d'))
                            ->where('end_date', '>=', date('Y-m-d'));
                    })
                    ->orWhere('permanent', 1);
            })->whereTranslation('status', 'published');
    }

    /**
     * Método público para la recuperación de páginas con restaurante
     *
     * @param  Site               $site   Variable de sitio
     * @return Builder                    Consulta filtrada
     */
    public function getRestaurantsBySite(Site $site)
    {
        return $this->has('restaurant')
            ->with('restaurant')
            ->where('site_id', $site->id)
            ->whereTranslation('status', 'published');
    }

    /**
     * Método público para la recuperación de páginas con habitación
     *
     * @param  Site               $site   Variable de sitio
     * @return Builder                    Consulta filtrada
     */
    public function getRoomsBySite(Site $site)
    {
        return $this->has('room')
            ->with('room')
            ->where('site_id', $site->id)
            ->whereTranslation('status', 'published');
    }

    /**
     * Método privado para actualizar los permalinks de un sitio completo
     *
     * ADVERTENCIA: Esta implementación es privada y no transaccional
     * para poder usar internamente y evitar transacciones anidadas
     *
     * @param  Site|null   $site Variable de sitio
     * @return int               Cantidad de registros afectados
     */
    private function rewritePermaLinks($site = null)
    {
        /** @var self Variable para mantener contexto en funciones anónimas */
        $self = $this;

        /** @var Collection|string[] Colección para guardar cambios a realizar, con llaves que representan el ID del objeto PageTranslation y el valor nuevo del campo "perman_link" */
        $permalink_changes = collect();

        /** @var callable Función recursiva para recuperar y crear los nuevos permalinks */
        $recursive_permalink_update = function ($value, $index) use (&$recursive_permalink_update, &$permalink_changes, $self) {

            /* Ciclar las traducciones */
            $value->translations->map(function ($value_translation, $index_translation) use ($value, &$permalink_changes, $self) {

                /* Si es raíz y no es home, generar los permalinks de acuerdo con el idioma */
                if ($value->isRoot() && !$self->isHome($value_translation->slug)) {
                    /* Estándar: Inglés es el lenguaje predeterminado */
                    $language = ($value_translation->locale == 'en')
                        ? ''
                        : $value_translation->locale;

                    /* Añadir ID de traducción y nuevo permalink a lista de cambios */
                    $permalink_changes->put(
                        $value_translation->id,
                        /* Añadir "/" al inicio, si no existe */
                        Str::start(
                            implode([
                                $language,
                                /* Quitar espacios y diagonales al principio y fin */
                                trim($value_translation->slug, " /")
                            ], '/'),
                            '/'
                        )
                    );
                } elseif (!$value->isRoot()) {
                    /* Obtener la traducción del padre */
                    $parent_translation = $value->parent->translate($value_translation->locale);

                    /* Cuando el padre no tiene una traducción en el mismo idioma, tomar la primera disponible (siempre hay al menos una) */
                    if (is_null($parent_translation)) {
                        $parent_translation = $value->parent->translations->first();
                    }

                    /* Si ya hay un permalink construido en $permalink_changes, tomar de ahí; si no, del padre */
                    $parent_permalink = ($permalink_changes->has($parent_translation->id))
                        ? $permalink_changes->get($parent_translation->id)
                        : $parent_translation->perman_link;

                    /* Añadir ID de traducción y nuevo permalink a lista de cambios */
                    $permalink_changes->put(
                        $value_translation->id,
                        implode([
                            $parent_permalink,
                            /* Quitar espacios y diagonales al principio y fin */
                            trim($value_translation->slug, " /")
                        ], '/')
                    );
                }
            });

            /* Si el modelo tiene hijos, aplicar de forma recursiva */
            if (!is_null($value->children)) {
                $value->children->map($recursive_permalink_update);
            }
        };

        /* Aplicar función recursiva a árbol de páginas de sitio solicitado */
        $this->toTrees((is_null($site)) ? Site::current() : $site)->map($recursive_permalink_update);

        /* Para este momento, $permalink_changes ya tiene todas las IDs de traducciones y los nuevos permalinks. Necesario construir consulta cruda para evitar ciclo con múltiples llamadas y retornar las filas afectadas */
        return PageTranslation::query()
            ->setBindings($permalink_changes->mapWithKeys(function ($value, $index) {
                return ["id{$index}" => $value];
            })->all())
            ->update([
                'perman_link' => DB::raw(
                    implode([
                        'CASE `id`',
                        $permalink_changes->map(function ($value, $index) {
                            return "WHEN {$index} THEN :id{$index}";
                        })->implode(' '),
                        'ELSE `perman_link` END'
                    ], ' ')
                )
            ]);
    }

    /**
     * Método protegido para ejecutar el comando de actualización de
     * galerías dinámicas de acuerdo con cambios en los datos.
     *
     * ADVERTENCIA: No ejecutar dentro de transacción, pues el comando
     * ya está envuelto en una.
     *
     * Los criterios con los cuales ejecuta son (todos deben cumplirse):
     *  - Si $original no es null y $model cambia desde o hacia
     *    estado "published"
     *  - Si $original es null y $model ya tiene estado "published",
     *
     * Si el modelo es guardado como "published" cuando ya era
     * originalmente el mismo tipo, la ejecución no sucederá.
     *
     * Este método no se encarga de filtrar todas las fechas o hacer
     * los cambios. Sólo se encarga de diferenciar cuándo llamar al
     * comando y cuándo no.
     *
     * @param  PageTranslation  $model        El modelo a cambiar
     * @param  array|null       $original     Arreglo con los datos originales
     *                                        Pred.: null
     * @param  string           $language     Identificador de localización
     *                                        (Ej.: "es", "en"...)
     * @return void
     * @throws Exception
     */
    protected function triggerDynamicGalleriesUpdate($model, $original = null)
    {
        try {
            /* Valores de nuevo y anterior */
            $isPublished = ($model->status == 'published');
            $wasPublished = (is_null($original))
                /* Para circunventar el "||" exclusivo si $model es "published" */
                ? false
                : ($original['status'] == 'published' && ($original['status'] != $model->status));

            /*
                "||" exclusivo (XOR): Sólo hacer cambios cuando se
                está transicionando a "header"

                Funciona igual que lo siguiente (los paréntesis son importantes):

                $togglePublished = ($wasPublished xor $isPublished);
             */
            $togglePublished = ($wasPublished ||  $isPublished) && !($wasPublished &&  $isPublished);

            /* Solamente cambiar si el valor se está cambiando a o desde "published" */
            if ($togglePublished) {
                /*
                    Ejecutar la actualización si una de las relaciones no está vacía.

                    Este comando se encargará de revisar fechas y demás.
                */
                if ($model->page->relationshipsWithFeaturables()->isNotEmpty()) {
                    $callResult = Artisan::call('mgcms2019:update-dynamic-gallery-priorities');

                    if ($callResult != 0) {
                        throw new Exception("No pudo ejecutarse el comando de actualización de galerías dinámicas. La consola arrojó lo siguiente: " . Artisan::output());
                    }
                }
            }
        } catch (Exception $e) {
            /* Volver a arrojar excepción */
            throw $e;
        }
    }
}
