<?php

namespace AvengersMG\MGCms2019\App\Cms\Pages;

use \Exception;
use Astrotomic\Translatable\Translatable;
use AvengersMG\MGCms2019\App\Cms\Accommodations\Room;
use AvengersMG\MGCms2019\App\Cms\Components\Cards\Card;
use AvengersMG\MGCms2019\App\Cms\Components\Forms\Form;
use AvengersMG\MGCms2019\App\Cms\Components\Galleries\Gallery;
use AvengersMG\MGCms2019\App\Cms\Components\Galleries\GalleryFeature;
use AvengersMG\MGCms2019\App\Cms\Components\Specials\Special;
use AvengersMG\MGCms2019\App\Cms\Components\Tours\Tour;
use AvengersMG\MGCms2019\App\Cms\Components\Restaurants\Restaurant;
use AvengersMG\MGCms2019\App\Cms\Mediafiles\FileType;
use AvengersMG\MGCms2019\App\Cms\Mediafiles\Mediafile;
use AvengersMG\MGCms2019\App\Cms\Pages\PageTranslation;
use AvengersMG\MGCms2019\App\Cms\Sites\Site;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Collection;
use Kalnoy\Nestedset\NodeTrait;

class Page extends Model
{
    use SoftDeletes;
    use Translatable;
    use NodeTrait;

    protected $table = 'pages';

    public $translatedAttributes = ['name', 'title', 'sub_title', 'perman_link', 'description', 'excerpt', 'status', 'navigation', 'slug', 'meta_title', 'meta_description', 'seo', 'locale'];
    protected $fillable = ['layout', 'template', '_lft', '_rgt', 'parent_id', 'status', 'site_id', 'form_id', 'index', 'is_roomprices_list'];

    protected $dates = ['deleted_at'];

    public function site()
    {
        return $this->belongsTo(Site::class);
    }

    public function mediafiles()
    {
        return $this->morphMany(Mediafile::class, 'mediafilable');
    }

    public function room()
    {
        return $this->hasOne(Room::class);
    }

    public function special()
    {
        return $this->hasOne(Special::class);
    }

    public function tour()
    {
        return $this->hasOne(Tour::class);
    }

    public function cards()
    {
        return $this->hasMany(Card::class);
    }

    public function restaurant()
    {
        return $this->hasOne(Restaurant::class);
    }

    public function pageRelations()
    {
        return $this->hasMany(Relation::class);
    }

    public function galleries()
    {
        return $this->hasMany(Gallery::class);
    }

    /**
     * Método público para establecer la relación entre la página y las fuentes
     * de información para las galerías propias actuales
     *
     * @return HasManyThrough Relación de pertenencia
     */
    public function gallerySources()
    {
        return $this->hasManyThrough(GalleryFeature::class, Gallery::class);
    }


    public function card($card)
    {
        return $this->cards->where('slug', $card)->first();
    }

    public function getRelatedPage($name)
    {
        $page = $this->pageRelations->where('name', $name)->first();
        if ($page) {
            return $page = Page::find($page->related_page);
        } else {
            return false;
        }
    }

    public function getPrimaryAttribute()
    {
        $exp = explode('|', $this->title);
        return $exp[0];
    }

    public function getSecondaryAttribute()
    {
        $exp = explode('|', $this->title);

        if (count($exp) > 1) {
            return $exp[1];
        }
        return '';
    }

    public function getFullTitleAttribute()
    {
        $exp = explode('|', $this->title);
        $title = '';

        foreach ($exp as $item) {
            $title = $title.$item. ' ';
        }

        return $title;
    }


    public function getChildren()
    {
        return $this->children;
    }

    /**
     * Método público para recuperar todos los archivos de medio que corresponden al modelo, de acuerdo con el tipo
     * @param  string         $type       Identificador de tipo de archivo
     * @param  string|null    $lang       Identificador de localización
     *                                    ("en", "es"...)
     * @param  boolean        $imagesOnly Bandera para indicar si limitar
     *                                    a imágenes. Pred.: false
     * @return Collection|Mediafile[]     Los modelos
     */
    public function getMedia($type, $lang = null, $imagesOnly = false)
    {
        return $this->retrieveMediafilesByFiletypeTag($type, $lang, $imagesOnly)->get();
    }

    public function getGallery($name)
    {
        $gallery = $this->galleries()->where('name', $name)->with('items.mediafile.translations')->first();

        if ($gallery) {
            return $gallery->items->sortBy('priority')->pluck('mediafile');
        }

        /* No encontró la galería */
        return [];
    }

    /**
     * Método público para recuperar el primer tipo de objeto Mediafile que este modelo tenga asignado
     *
     * 2019-05-08: Retorno normalizado
     *
     * @param  string         $type       Etiqueta de tipo de Mediafile
     * @param  string|null    $lang       Identificador de internacionalización
     *                                    ("en", "es"...)
     * @param  boolean        $imagesOnly Bandera para indicar si limitar
     *                                    a imágenes. Pred.: false
     * @return Mediafile|null       Resultado de la búsqueda
     */
    public function getFirst($type, $lang = null, $imagesOnly = false)
    {
        return $this->retrieveMediafilesByFiletypeTag($type, $lang, $imagesOnly)->first();
    }

    /**
     * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
     * Relacion con Wysiwyg
     */

    public function content()
    {
        return $this->belongsToMany(Wysiwyg::class);
    }

    public function getContent($content)
    {
        return $this->content->where('name', $content)->first()->wysiwyg;
    }

    /**
     * Método público para crear ámbito para recuperar objetos por objeto Site
     *
     * @param  Builder                   $query   Consulta inyectada
     * @param  Collection|Site[]|Site    $params  Sitio(s) para filtrar
     * @return Builder                            La consulta fitrada
     * @throws Exception
     */
    public function scopeBySite($query, $params)
    {
        /* Envolver en colección, si es que no viene en una */
        $sites = Collection::wrap($params);

        return $query->whereIn('site_id', $sites->pluck('id')->all());
    }

    /**
     * Mutador para establecer los valores permitidos al editar la propiedad
     * is_roomprices_list en el modelo
     *
     * @param mixed $value Valor enviado para escribir
     */
    public function setIsRoompricesListAttribute($value)
    {
        /* Sólo permitir los valores true y null */
        $parsedValue = (((boolean) $value) == true)
            ? true
            : null;

        $this->attributes['is_roomprices_list'] = $parsedValue;
    }

    /**
     * Método público para buscar relaciones no vacías de la instancia
     * con clases que tengan la interfaz "FeaturableInterface"
     *
     * @return Collection|string[] Arreglo de métodos de relaciones
     */
    public function relationshipsWithFeaturables()
    {
        /*
           Nombres de métodos de las fuentes especificadas por las galerías
           del sitio al que esta instancia pertenece y en la que la relación
           no está vacía
        */
        return GalleryFeature::whereHas('gallery.page', function ($query) {
            $query->where('site_id', $this->site_id);
        })->get()->map(function ($feature) {
            return call_user_func("{$feature->featurable_type}::getFeaturableRelationshipMethod");
        })->unique()->filter(function ($methodName) {
            /* Nota: inevitable este bucle sin hacer reflexión */
            return call_user_func([$this, $methodName])->count() > 0;
        });
    }

    /**
     * Método protedigo para filtrar los mediafiles, de acuerdo con el tipo
     *
     * @param  string      $type       Etiqueta de tipo de Mediafile
     * @param  string|null $lang       Identificador de internacionalización
     *                                 ("es", "en"...)
     * @param  boolean     $imagesOnly Bandera para indicar si limitar
     *                                 a imágenes. Pred.: false
     * @return Builder      La consulta fitrada
     */
    protected function retrieveMediafilesByFiletypeTag($type, $lang = null, $imagesOnly = false)
    {
        if (is_null($lang)) {
            $lang = app()->getLocale();
        }

        /* Definir cuándo usar sólo imágenes */
        $source = ($imagesOnly)
            ? $this->mediafiles()->imagesOnly()
            : $this->mediafiles();

        return $source->whereHas('type', function($query) use ($type){
            $query->where('type', $type);
        })->translatedIn($lang);
    }
}
