<?php

namespace AvengersMG\MGCms2019\App\Cms\Mediafiles\Repositories;

use \Exception;
use AvengersMG\MGCms2019\App\Cms\BaseRepository\BaseRepository;
use AvengersMG\MGCms2019\App\Cms\Components\Galleries\GalleryFeature;
use AvengersMG\MGCms2019\App\Cms\Mediafiles\FileType;
use AvengersMG\MGCms2019\App\Cms\Mediafiles\Mediafile;
use AvengersMG\MGCms2019\App\Cms\Mediafiles\Repositories\MediafilesInterface;
use AvengersMG\MGCms2019\App\Cms\Pages\Page;
use AvengersMG\MGCms2019\App\Cms\Sites\Site;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class MediaRepository extends BaseRepository implements MediafilesInterface
{
    public function __construct()
    {
        parent::__construct(Mediafile::class);
    }

    public function withType($id)
    {
        return call_user_func("{$this->model}::with", 'type')->where('id', $id)->first();
    }

    /**
     * Método público para recuperar un objeto Mediafile de acuerdo con su dirección
     *
     * @param  string     $path Dirección del archivo que representa
     * @param  Site       $site El sitio al que pertenece
     * @return Mediafile
     */
    public function findWithPath($path, Site $site)
    {
        return call_user_func("{$this->model}::bySite", $site)->where('path', $path)->first();
    }

    /**
     * Método público para recuperar una colección de objetos Mediafile
     * de acuerdo con un arreglo de direcciones
     *
     * @param  string[]               $paths  Arreglo de direcciones a buscar
     * @param  Site                   $site   El sitio al que pertenece
     * @return Collection|Mediafile[]
     */
    public function findWithPaths($paths, Site $site)
    {
        return call_user_func("{$this->model}::bySite", $site)->whereIn('path', $paths)->with('translations')->get();
    }

    /**
     * Método público para modificar las propiedades de un objeto MediaFile
     * y renombrar el archivo al que apunta
     *
     * @param  Request                          $request    Información de la solicitud
     * @param  Mediafile                        $mediafile  Objeto a modificar
     * @param  string|null                      $uploadDisk Identificador de disco
     * @return MediaFile                                    Objeto MediaFile modificado
     * @throws ModelNotFoundException|Exception
     */
    public function updateAndRenameFileByRequest(Request $request, $mediafile, $uploadDisk = null)
    {
        $updatingResult = false;

        /**
         * Espacio para información de modelo original (antes de cambio)
         *
         * @var array
         */
        $original = [];

        /*
          Asegurar que la etiqueta de disco sea la predeterminada
          si es que no fue especificada
        */
        if (is_null($uploadDisk)) {
            $uploadDisk = $this->uploadDisk;
        }

        /* Obtener nuevo nombre de archivo */
        $newFileName = $this->slugifyFileName($request->name);

        /*
          Reemplazar el nombre antiguo por el nuevo
          ($request->path tiene al final el nombre antiguo de archivo)
         */
        $newFilePath = preg_replace('/([^\/]+)$/', $newFileName, $request->path);

        /* Iniciar transacción */
        DB::beginTransaction();

        try {
            $file = $mediafile;

            /* Obtener información de modelo original */
            $original = $file->getOriginal();

            /* Hacer los cambios a la base de datos antes de mover el sistema de archivos */
            $file->name = $newFileName;
            $file->path = $newFilePath;
            $file->type_id = $request->type_id;
            $file->status = $request->status;

            /* Guardar el modelo */
            $updatingResult = $file->save();

            /* Recuperar la traducción */
            $file_translation = $file->translateOrNew($request->locale);

            /* Asignar ID de modelo */
            if (empty($file_translation->mediafile_id)) {
                $file_translation->mediafile_id = $file->id;
            }

            /* Traducibles */
            $file_translation->title = $request->title;
            $file_translation->description = $request->description;
            if ($request->has('alt')) {
                $file_translation->alt = $request->alt;
            }
            if ($request->has('link')) {
                $file_translation->link = $request->link;
            }

            /* Guardar la traducción */
            $file_translation->save();

            /* Guardado exitoso: Buscar archivo y moverlo de lugar si es que los nombres son diferentes  */
            if (Storage::disk($uploadDisk)->exists($request->path) && ($newFilePath != $request->path)) {
                /* Si no logra cambiar el archivo, arrojar nueva excepción */
                if (!Storage::disk($uploadDisk)->move($request->path, $newFilePath)) {
                    throw new Exception('No fue posible cambiar nombre a archivo');
                }
            }

            /* Proceso terminó correctamente. Aplicar cambios */
            DB::commit();
        } catch (Exception $e) {
            /* Proceso falló. Deshacer cambios en la base de datos */
            DB::rollback();

            /* Regresar al valor negativo */
            $updatingResult = false;

            /* Volver a arrojar la excepción */
            throw $e;
        }

        /* Si la actualización fue exitosa, ejecutar el comando de revisión */
        if ($updatingResult == true) {
            /*
                $file todavía contiene la información del modelo e,
                incluso, puede cargar sus relaciones aunque ya no persista en la
                base de datos
            */
            $this->triggerDynamicGalleriesUpdate($file, $original);
        }

        /* Todo salió bien. Retornar archivo */
        return $file;
    }

    /**
     * Método público para la destrucción de un modelo MediaFile,
     * junto con la eliminación del archivo al que apunta
     *
     * @param  Mediafile    $mediafile  Objeto a modificar
     * @param  string|null  $uploadDisk Identificador de disco
     * @return void
     * @throws ModelNotFoundException|Exception
     */
    public function deleteAndRemoveFile($mediafile, $uploadDisk = null)
    {
        $deletionResult = false;

        /*
          Asegurar que la etiqueta de disco sea la predeterminada
          si es que no fue especificada
        */
        if (is_null($uploadDisk)) {
            $uploadDisk = $this->uploadDisk;
        }

        /* Iniciar transacción */
        DB::beginTransaction();

        try {
            $file = $mediafile;
            /* Eliminar registro antes de mover sistema de archivos. Las traducciones cascadean */
            $deletionResult = $file->delete();

            /* Si no se logra eliminar, arrojar excepción */
            if (!Storage::disk($uploadDisk)->delete($file->path)) {
                throw new Exception('No fue posible eliminar archivo');
            }

            /* Proceso terminó correctamente. Aplicar cambios */
            DB::commit();
        } catch (Exception $e) {
            /* Proceso falló. Deshacer cambios en la base de datos */
            DB::rollback();

            /* Regresar al valor negativo */
            $deletionResult = false;

            /* Volver a arrojar la excepción */
            throw $e;
        }

        /* Si la eliminación fue exitosa, ejecutar el comando de revisión */
        if ($deletionResult == true) {
            /*
                $mediafile todavía contiene la información del modelo e,
                incluso, puede cargar sus relaciones aunque ya no persista en la
                base de datos
            */
            $this->triggerDynamicGalleriesUpdate($mediafile);
        }
    }

    /**
     * Método para crear un nuevo objeto MediaFile para un sitio o una página, por medio de Request
     *
     * @param  Request                          $request      Información de la solicitud
     * @param  string|null                      $uploadDisk   Identificador de disco
     * @param  Model|null                       $mediafilable Modelo al qué asignar
     *                                                        el resultado
     * @return MediaFile                                      El objeto creado
     * @throws ModelNotFoundException|Exception
     */
    public function createWithFileByRequest(Request $request, $uploadDisk = null, $mediafilable = null)
    {
        /*
          Asegurar que la etiqueta de disco sea la predeterminada
          si es que no fue especificada
        */
        if (is_null($uploadDisk)) {
            $uploadDisk = $this->uploadDisk;
        }

        /* Obtener la internacionalización del sistema */
        $lang = app()->getLocale();

        /* Obtener archivo para usos futuros */
        $requestFile = $request->file('file');

        /* Obtener tamaño y MIME */
        $fileSize = $requestFile->getSize();
        $mimeType = $requestFile->getClientMimeType();

        /* Iniciar transacción */
        DB::beginTransaction();

        try {
            /* Establecer mediafilable si no existe */
            if (!is_a($mediafilable, Model::class, false)) {
                /* Seleccionar a qué objeto debe asignarse el resultado */
                if ($request->has('page_id')) {
                    /* Asignar a página */
                    $mediafilable = Page::findOrFail($request->page_id);
                } elseif ($request->has('site_id')) {
                    /* Asignar a sitio específico */
                    $mediafilable = Site::findOrFail($request->site_id);
                } else {
                    /* Asignar a sitio actual */
                    $mediafilable = Site::current();

                    if (is_null($mediafilable)) {
                        throw new Exception('No existen sitios creados');
                    }
                }
            }

            /* Guardar archivo en ubicación definitiva (sin sobrescribir) */
            $upload = $this->uploadFileByRequest($request, $request->path, false, $uploadDisk);

            /* Guardar actividades en log */
            Log::info($request->getHttpHost());

            /** @var mixed Obtener respuesta o MediaFile creado */
            $response = $this->create([
                'name' => $upload->get('filename'),
                'path' => '/'.$upload->get('path'),
                'file_size' => $fileSize,
                'content_type' => $mimeType,
                "{$lang}" => [
                    'title' => null,
                ],
                'mediafilable_id' => $mediafilable->id,
                'mediafilable_type'=> get_class($mediafilable),
            ]);

            /* Todo terminó correctamente. Aplicar cambios */
            DB::commit();
        } catch (Exception $e) {
            /* Proceso falló. Deshacer cambios en la base de datos */
            DB::rollback();

            /* Volver a arrojar la excepción */
            throw $e;
        }

        /* Si la creación fue exitosa, ejecutar el comando de revisión */
        if ($response instanceof Mediafile) {
            $this->triggerDynamicGalleriesUpdate($response);
        }

        /* Todo terminó correctamente, retornar resultado */
        return $response;
    }

    /**
     * Método público para el muestreo de archivos en un directorio
     *
     * @param  string       $dir         Dirección
     * @param  Site         $site        Objeto Site para referencia
     * @param  string|null  $uploadDisk  Identificador de disco
     * @return array[]                   Arreglo asociativo con arreglo de archivos ('files')
     *                                   y directorios ('dirs')
     */
    public function listfiles($dir, Site $site, $uploadDisk = null)
    {
        /* Variable para mantener contexto */
        $self = $this;

        /*
          Asegurar que la etiqueta de disco sea la predeterminada
          si es que no fue especificada
        */
        if (is_null($uploadDisk)) {
            $uploadDisk = $this->uploadDisk;
        }

        /* Si no existe, crear */
        if (!Storage::disk($uploadDisk)->exists($dir)) {
            Storage::disk($uploadDisk)->makeDirectory($dir);
        }

        /* Crear colección de direcciones de archivos */
        $filesPath = collect(Storage::disk($uploadDisk)->files($dir))->map(function ($item) {
            return [
                'basename' => basename($item), /* Nombre de archivo, con extensión */
                'path' => Str::start($item, '/'),
            ];
        });

        /* Obtener los objetos que coinciden con las direcciones */
        $mediaFilesCollection = $this->findWithPaths($filesPath->pluck('path')->all(), $site);

        /* Recuperar nombres de archivos y modelos comparándolos con los resultados de la búsqueda por sitio  */
        $files = $filesPath->map(function ($item, $key) use ($mediaFilesCollection) {
            /* Envolver en colección */
            $collectedItem = collect($item);
            /* Tratar de recuperar modelo */
            if ($model = $mediaFilesCollection->where('path', $collectedItem->get('path'))->first()) {
                /* Retornar arreglo */
                return [
                    'basename' => $collectedItem->get('basename'),
                    'model' => $model,
                ];
            }

            /* Modelo no encontrado: Filtro no debe de pasar */
            return false;
        })->filter()->all();

        /* Recuperar directorios */
        $dirs = collect(Storage::disk($uploadDisk)->directories($dir))->map(function ($item, $key) {
            /* Recuperar última parte del URL */
            return basename($item);
        })->all();

        /* Retornar datos encontrados */
        return [
            'files' => $files,
            'dirs' => $dirs
        ];
    }

    /**
     * Método público para la creación de directorios detro del disco
     * @param  Request       $request    Información de solicitud
     * @param  string|null   $uploadDisk Identificador de disco
     * @return boolean                   Respuesta de creación de directorio
     */
    public function createDirectoryByRequest(Request $request, $uploadDisk = null)
    {
        /*
          Asegurar que la etiqueta de disco sea la predeterminada
          si es que no fue especificada
        */
        if (is_null($uploadDisk)) {
            $uploadDisk = $this->uploadDisk;
        }

        return Storage::disk($uploadDisk)->makeDirectory($request->path.'/'.$request->name);
    }

    /**
     * Método público para verificar que un directorio existe en el disco
     *
     * @param  string       $path       Dirección del directorio
     * @param  string|null  $uploadDisk Identificador de disco
     * @return boolean                  Resultado de revisión
     */
    public function isDirectory($path, $uploadDisk = null)
    {
        /*
          Asegurar que la etiqueta de disco sea la predeterminada
          si es que no fue especificada
        */
        if (is_null($uploadDisk)) {
            $uploadDisk = $this->uploadDisk;
        }

        /* Retornar resultado de la búsqueda */
        return Storage::disk($uploadDisk)->exists($path);
    }

    /**
     * Método protegido para ejecutar el comando de actualización de
     * galerías dinámicas de acuerdo con cambios en los datos.
     *
     * ADVERTENCIA: No ejecutar dentro de transacción, pues el comando
     * ya está envuelto en una.
     *
     * Los criterios con los cuales ejecuta son (todos deben cumplirse):
     *  - $model es una imagen
     *  - Si $original no es null y $model cambia desde o hacia tipo "header"
     *    o "header_mobile",
     *  - Si $original es null y $model ya es de tipo "header" o "header_mobile",
     *  - $model pertenece directamente a una página (no a un sitio),
     *  - La página asociada tiene una relación no nula a un objeto con
     *    interfaz "featurable", cuya clase influencía al menos a una galería.
     *
     * Si el modelo es guardado como "header" o "header_mobile" cuando ya era
     * originalmente el mismo tipo, la ejecución no sucederá.
     *
     * Este método no se encarga de filtrar todas las fechas o hacer
     * los cambios. Sólo se encarga de diferenciar cuándo llamar al
     * comando y cuándo no.
     *
     * @param  Mediafile  $model        El modelo a cambiar
     * @param  array|null $original     Arreglo con los datos originales
     *                                  Pred.: null
     * @return void
     * @throws Exception
     */
    protected function triggerDynamicGalleriesUpdate($model, $original = null)
    {
        /* Solo ejecutar si es una imagen */
        if ($model->is_image) {
            try {
                /* Tiene que cargarse tiempo de ejecución y no en el booteo */
                $headerFileTypes = FileType::headers()->pluck('id');

                /* Valores de nuevo y anterior */
                $isHeader = ($headerFileTypes->contains($model->type_id));
                $wasHeader = (is_null($original))
                    /* Para circunventar el "||" exclusivo si $model es "header" o "header_mobile" */
                    ? false
                    : ($headerFileTypes->contains($original['type_id']) && ($original['type_id'] != $model->type_id));

                /*
                    "||" exclusivo (XOR): Sólo hacer cambios cuando se
                    está transicionando a "header"

                    Funciona igual que lo siguiente (los paréntesis son importantes):

                    $toggleHeader = ($wasHeader xor $isHeader);
                 */
                $toggleHeader = ($wasHeader ||  $isHeader) && !($wasHeader &&  $isHeader);

                /* Solamente cambiar si el valor se está cambiando a o desde "header"/"header_mobile" */
                if ($toggleHeader) {
                    /**
                     * Todas las fuentes que podrían con el cambio de este mediafile
                     *
                     * @var Collection|GalleryFeature[]
                     */
                    $featureSources = GalleryFeature::pluck('featurable_type')->unique();

                    /* El padre del mediafile: Generalmente una página o un sitio */
                    $mediafilable = $model->mediafilable;

                    /*
                        Un mediafile sólo puede afectar las galerías dinámicas si es
                        propiedad de una página
                    */
                    if ($mediafilable instanceof Page) {
                        /** @var Collection|string[] Nombres de métodos */
                        $featureRelationsMethods = $featureSources->map(function ($sourceClass) {
                            return call_user_func("{$sourceClass}::getFeaturableRelationshipMethod");
                        });

                        /*
                          Cargar todos los modelos necesarios de auerdo con las fuentes
                          de imágenes.
                        */
                        $mediafilable->load($featureRelationsMethods->all());

                        /** @var Collection|mixed[] Resultados de búsqueda de objetos */
                        $mediafilableFeatures = $featureRelationsMethods->map(function ($methodName) use ($mediafilable){
                            return $mediafilable->{$methodName};
                        })->filter();

                        /*
                            Ejecutar la actualización si no está vacía.

                            Este comando se encargará de revisar fechas y demás.
                        */
                        if ($mediafilableFeatures->isNotEmpty()) {
                            $callResult = Artisan::call('mgcms2019:update-dynamic-gallery-priorities');

                            if($callResult != 0){
                                throw new Exception("No pudo ejecutarse el comando de actualización de galerías dinámicas. La consola arrojó lo siguiente: " . Artisan::output());
                            }
                        }
                    }
                }
            } catch (Exception $e) {
                  /* Volver a arrojar excepción */
                 throw $e;
            }
        }
    }
}
