<?php

namespace AvengersMG\MGCms2019\App\Cms\Mediafiles;

use Astrotomic\Translatable\Translatable;
use AvengersMG\MGCms2019\App\Cms\Components\Galleries\Item;
use AvengersMG\MGCms2019\App\Cms\Mediafiles\FileType;
use AvengersMG\MGCms2019\App\Cms\Pages\Page;
use AvengersMG\MGCms2019\App\Cms\Sites\Site;
use AvengersMG\MGCms2019\App\Cms\Seo\Seo;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Str;

class Mediafile extends Model
{
    use Translatable;
    /* Establecerá funciones en los eventos de Eloquent */

    protected $table = 'mediafiles';
    protected $fillable = ['mediafilable_id', 'mediafilable_type','priority', 'name', 'type_id', 'path', 'status', 'file_size', 'content_type'];
    protected $orderBy="priority";
    protected $appends = ['is_image', 'file_extension'];
    public $translatedAttributes = ['link', 'title', 'description', 'alt'];

    /*
        Extensiones para validar solo imágenes.
        "static" para utilizarlo en ámbitos
    */
    protected static $imageExtensions = [
        'jpg',
        'jpeg',
        'jpe',
        'jif',
        'jfif',
        'jfi',
        'png',
        'gif',
        'webp',
        'tiff',
        'tif',
        'psd',
        'raw',
        'arw',
        'cr2',
        'nrw',
        'k25',
        'bmp',
        'dib',
        'heif',
        'heic',
        'ind',
        'indd',
        'indt',
        'jp2',
        'j2k',
        'jpf',
        'jpx',
        'jpm',
        'mj2',
        'svg',
        'svgz',
        'ai',
        'eps',
        'pdf',
    ];

    /* Sobrescritura de método de booteo */
    public static function boot()
    {
        /* Ejecutar el booteo regular */
        parent::boot();

        /* Añadir scope global */
        static::addGlobalScope('order', function (Builder $builder) {
            $builder->orderBy('priority', 'desc');
        });
    }

    public function mediafilable()
    {
        return $this->morphTo();
    }

    public function type()
    {
        return $this->belongsTo(FileType::class, 'type_id');
    }

    public function getFileName($path)
    {
        return basename($path);
    }

    /**
     * Método público para emular el sitio al que el objeto pertenece
     * @return Site El sitio
     */
    public function getSiteAttribute()
    {
        $owner = $this->mediafilable;
        return ($owner instanceof Site)? $owner : $owner->site;
    }

    /**
     * Método público para crear ámbito para recuperar objetos por objeto Site
     *
     * @param  Builder $query Consulta inyectada
     * @param  Site    $site  Sitio para filtrar
     * @return Builder        La consulta fitrada
     */
    public function scopeBySite($query, Site $site)
    {
        /* Para limitar el ámbito */
        return $query->where(function ($query) use ($site) {
            $query->where([
                ['mediafilable_type', '=', Site::class],
                ['mediafilable_id', '=', $site->id]
            ])->orWhere(function ($query) use ($site) {
                $query->whereIn('mediafilable_id', Page::where('site_id', $site->id)->pluck('id')->all())->where('mediafilable_type', Page::class);
            });
        });
    }

    /**
     * Método público para emular propiedad que indica si el objeto es imagen
     *
     * @return boolean Bandera de indicación
     */
    public function getIsImageAttribute()
    {
        if (is_string($this->content_type)) {
            /* Revisar "content_type" */
            return Str::startsWith($this->content_type, 'image/');
        }

        /* "content_type" no está configurado. Revisar extensión */
        return Str::endsWith($this->name, collect(static::$imageExtensions)->map(function ($value) {
            return ".{$value}";
        })->all());
    }

    /**
     * Método público para emular propiedad que indica la extensión del archivo
     *
     * @return string Extensión
     */
    public function getFileExtensionAttribute()
    {
        return pathinfo($this->name, PATHINFO_EXTENSION);
    }

    /**
     * Método público para crear ámbito para recuperar solo objetos que
     * representan imágenes
     *
     * @param  Builder $query Consulta inyectada
     * @return Builder        La consulta fitrada
     */
    public function scopeImagesOnly($query)
    {
        /* Para limitar el ámbito */
        return $query->where(function ($query) {
            /* Sólo los que tengan "content_type" como imagen */
            $query->where('content_type', 'LIKE', 'image/%');

            /*
                Condiciones adicionales para cuando "content_type" es NULL
                (por versiones anteriores del CMS)
             */
            foreach(static::$imageExtensions as $imageExtension) {
                $query->orWhere('name', 'LIKE', "%.{$imageExtension}");
            }
        });
    }

    /**
     * Método para establecer la relación entre el mediafile y los items que
     * lo utilizan
     *
     * @return HasMany Relación de pertenencia
     */
    public function items()
    {
        return $this->hasMany(Item::class);
    }
}
