<?php
/**
 * Created by PhpStorm.
 * User: MAC_82_MKT_Full_Stack
 * Date: 17/12/18
 * Time: 12:14
 */

namespace AvengersMG\MGCms2019\App\Cms\Components\Specials;

use \Exception;
use AvengersMG\MGCms2019\App\Cms\BaseRepository\BaseRepository;
use AvengersMG\MGCms2019\App\Cms\Components\Galleries\GalleryFeature;
use AvengersMG\MGCms2019\App\Cms\Pages\Page;
use AvengersMG\MGCms2019\App\Cms\Sites\Site;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;
use AvengersMG\MGCms2019\App\Events\SpecialEvent;
use App\Notifications\PushSpecials;
use AvengersMG\MGCms2019\App\Cms\Users\Guest;
use Notification;

class SpecialRepository extends BaseRepository implements SpecialInterface
{
    public function __construct()
    {
        parent::__construct(Special::class);
    }

    /**
     * Método público para retornar todos los especiales de acuerdo con sitio,
     * parámetros múltiples de sitios o colección de sitios
     *
     * @param  Collection|Site[]|Site  $params  Variables de sitios: colección,
     *                                          múltiples sitios
     *                                          o único sitio
     * @return Builder                          Consulta filtrada
     * @throws Exception                        Si la entradas no coinciden con
     *                                          formatos esperados
     *
     */
    public function getActivesBySite($params)
    {
        /* Envolver en colección */
        $sites = Collection::wrap($params);

        /* Verificar cantidad para acción */
        if ($sites->count() > 1) {
            /** @var Collection|string[] Clases de objetos */
            $dataTypes = $sites->map(function ($value) {
                return get_class($value);
            })->unique()->values();

            /* Verificar datos enviados */
            if ($dataTypes->count() > 1) {
                throw new Exception("Más de un tipo de objeto. Solo debe haber un tipo");
            } else if ($dataTypes->first() != Site::class) {
                throw new Exception("Colección de objetos de tipo equivocado. Esperando colección de objetos tipo Site");
            }
        } else if ($sites->count() == 1) {
            if (!($sites->first() instanceof Site)) {
                throw new Exception("Parámetro único de tipo incorrecto. Esperando clase Site");
            }
        }

        /* Datos correctos. Aplicar para filtrado  */

        return $this->with('page')
            ->whereIn('page_id', Page::bySite($sites)->pluck('id')->all())
            ->whereHas('page', function ($pageQuery) {
                $pageQuery->whereTranslation('status', 'published');
            })
            ->where(function ($statusQuery) {
                $statusQuery->where(function ($query) {
                    $query
                        ->where('start_date', '<=', date('Y-m-d'))
                        ->where('end_date', '>=', date('Y-m-d'));
                })
                ->orWhere('permanent', 1);
            });
    }

    public function getDate($dates)
    {
        return explode(' ', $dates);
    }

    /**
     * Método público para crear un nuevo especial
     *
     * @param  Request $request La información proveniente del controlador
     * @return Special          Modelo creado
     */
    public function store($request)
    {
            app()->setLocale($request->locale);

          /* Fecha actual, para uso futuro */
          $now = Carbon::now();

          $booking_w = $this->getDate($request->booking_w);
          $travel_w = $this->getDate($request->travel_w);
        
          if(isset($request->bookingcta) || (isset($request->llamadacta) && isset($request->numbertel)) || (isset($request->linkcta) && isset($request->link))){
            $call_to_action = $this->createArray($request->bookingcta, $request->llamadacta, $request->numbertel, $request->numbertel_es, $request->linkcta, $request->namecta, $request->link, $request->namecta_es, $request->link_es);
         }

          /* Iniciar transacción */
          DB::beginTransaction();
  
          try {
              $special = [
                  'page_id' => $request->page_id,
                  'name' => $request->name,
                  'start_date' => date('Y-m-d H:i:s', strtotime($booking_w[0])),
                  'end_date' => date('Y-m-d H:i:s', strtotime($booking_w[2])),
                  'start_travel' => date('Y-m-d H:i:s', strtotime($travel_w[0])),
                  'end_travel' => date('Y-m-d H:i:s', strtotime($travel_w[2])),
                  'terms' => $request->terms,
                  'countdown' => $request->countdown,
                  'permanent' => $request->permanent,
                  'locale' => $request->locale,
                  'cta' => $call_to_action
              ];

              /* Obtener resultados de creación */
              $model = $this->create($special);

            
              if ($model instanceof Special) {
                  /* Traducción para uso futuro */
                  $translation = $model->translate();
  
                  /** @var Carbon Fecha y hora de notificación */
                  $notificationTime = Carbon::parse($request->notification_time);
  
                  /* Si no hay evento en el horario y la fecha es en el futuro,
                  crear nuevo evento */
                  if ($now->lessThanOrEqualTo($notificationTime)) {
                      /* Fecha en el futuro. Crear nuevo evento */
                      $model->translate()->notifications()->create([
                          'notification_time' => $notificationTime->toDateTimeString(),
                      ]);
                  }
  
                  $this->triggerDynamicGalleriesUpdate($model);
              }
  
              /* Aplicar cambios */
              DB::commit();
          } catch (Exception $e) {
              /* Deshacer cambios */
              DB::rollback();
  
              /* Retornar la excepción */
              throw $e;
          }
  
          /* Terminó sin problemas */
          return $model;
    }

    /**
     * Método público para actualizar un especial, de acuerdo con una petición
     *
     * @param  Request $request La información proveniente del controlador
     * @param  Special $model   El modelo a modificar
     * @return Special          Modelo modificado
     */
    public function updateSpecial($request, $model)
    {
        /* Fecha actual, para uso futuro */
        $now = Carbon::now();

        app()->setLocale($request->locale);

        $booking_w = $this->getDate($request->booking_w);
        $travel_w = $this->getDate($request->travel_w);

        if(isset($request->bookingcta) || isset($request->llamadacta) || (isset($request->linkcta) && isset($request->link))){
            $call_to_action = $this->createArray($request->bookingcta, $request->llamadacta, $request->numbertel, $request->numbertel_es, $request->linkcta, $request->namecta, $request->link, $request->namecta_es, $request->link_es);
        }

        /* Iniciar transacción */
        DB::beginTransaction();

        try {
            /* Obtener datos antes del cambio */
            $original = $model->getOriginal();
            $original['featurable_start_date'] = $model->featurable_start_date;
            $original['featurable_end_date'] = $model->featurable_end_date;

            $special = [
                'name' => $request->name,
                'start_date' => date('Y-m-d H:i:s', strtotime($booking_w[0])),
                'end_date' => date('Y-m-d H:i:s', strtotime($booking_w[2])),
                'start_travel' => date('Y-m-d H:i:s', strtotime($travel_w[0])),
                'end_travel' => date('Y-m-d H:i:s', strtotime($travel_w[2])),
                'terms' => $request->terms,
                'countdown' => ($request->countdown) ? $request->countdown : null ,
                'permanent' => ($request->permanent) ? $request->permanent : null ,
                'locale' => $request->locale,
                'cta' => $call_to_action
            ];

            $updateResult = $model->update($special);

            /* Recargar el modelo porque update() modifica el dato persistente */
            $model = $model->fresh();
            
            if ($updateResult == true) {
                /* Traducción para uso futuro */
                $translation = $model->translate($request->locale);

                $eventExists = $translation->notifications()->where('notification_time', $request->notification_time)->count() > 0;
                
                // return $request->notification_time;
                /** @var Carbon Fecha y hora de notificación */
                $notificationTime = Carbon::parse($request->notification_time);
                
                /* Si no hay evento en el horario y la fecha es en el futuro,
                crear nuevo evento */
                // $now->lessThanOrEqualTo($notificationTime solo permitía agregar fechas futuras
                if (!$eventExists) {
                    /* Fecha en el futuro. Crear nuevo evento */
                    $translation->notifications()->create([
                        'notification_time' => $notificationTime->toDateTimeString(),
                    ]);
                }
            }

            /* Realizar cambios */
            DB::commit();
        } catch (Exception $e) {
            /* Deshacer cambios */
            DB::rollback();

            /* Especificar que no se hizo la actualización */
            $updateResult = false;

            /* Arrojar excepción */
            throw $e;
        }

        /* Ejecutar revisión */
        if ($updateResult == true) {
            $this->triggerDynamicGalleriesUpdate($model, $original);
        }

        /* Retornar modelo cambiado */
        return $model;
    }

    /**
     * Método público para la eliminación de especial y sus relaciones
     *
     * @param  Special     $special Especial a eliminar
     * @return boolean              Resultado de eliminación
     */
    public function deleteAll($special)
    {
        $retorno = false;

        /* Iniciar transacción */
        DB::beginTransaction();

        try {
            /* Borrar traducciones y luego modelo */
            $special->deleteTranslations();
            $special->delete();

            /* Cambios realizados correctamente */
            DB::commit();

            /* Cambiar valor de retorno */
            $retorno = true;
        } catch (Exception $e) {
            /* Deshacer cambios */
            DB::rollback();
        }

        if ($retorno == true) {
            $this->triggerDynamicGalleriesUpdate($model);
        }

        return $retorno;
    }

    /**
     * Método protegido para ejecutar el comando de actualización de
     * galerías dinámicas de acuerdo con cambios en los datos.
     *
     * ADVERTENCIA: No ejecutar dentro de transacción, pues el comando
     * ya está envuelto en una.
     *
     * Los criterios con los cuales ejecuta son (todos deben cumplirse):
     *  - Si $original no es null y las fechas en $model cambian,
     *  - Si $original es null,
     *  - Si $model es de una clase que está solicitada como fuente en
     *   alguna galería.
     *
     * Este método no se encarga de filtrar todas las fechas o hacer
     * los cambios. Sólo se encarga de diferenciar cuándo llamar al
     * comando y cuándo no.
     *
     * @param  Special    $model        El modelo a cambiar
     * @param  array|null $original     Arreglo con los datos originales
     *                                  Pred.: null
     * @return void
     * @throws Exception
     */
    protected function triggerDynamicGalleriesUpdate($model, $original = null)
    {
        try {
            if (is_null($original)) {
                /* Si no existe el original, configurar para forzar cambio */
                $isDifferentStart = true;
                $isDifferentEnd = true;
                $togglePermanent = true;
            } else {
                /* Resultados de verificaciones de fechas */
                $isDifferentStart = ($model->featurable_start_date != $original['featurable_start_date']);

                $isDifferentEnd = ($model->featurable_end_date != $original['featurable_end_date']);

                /* Valores de nuevo y anterior */
                $togglePermanent = ($model->permanent != $original['permanent']);
            }

            /* Ejecutar si alguna de las fechas es diferente */
            if ($isDifferentStart || $isDifferentEnd || $togglePermanent) {
                /**
                 * Todas las fuentes que podrían con el cambio de este mediafile
                 *
                 * @var Collection|GalleryFeature[]
                 */
                $featureSources = GalleryFeature::pluck('featurable_type')->unique();

                /*
                    Ejecutar la actualización si está en las fuentes.

                    Este comando se encargará de revisar fechas y demás.
                */
                if ($featureSources->contains(get_class($model))) {
                    $callResult = Artisan::call('mgcms2019:update-dynamic-gallery-priorities');

                    if($callResult != 0){
                        throw new Exception("No pudo ejecutarse el comando de actualización de galerías dinámicas. La consola arrojó lo siguiente: " . Artisan::output());
                    }
                }
            }
        } catch (Exception $e) {
             /* Volver a arrojar excepción */
             throw $e;
        }
    }

    public function createSpecialEvent($special) {
        
            Notification::send(Guest::all(),new PushSpecials($special));
        /**
         * implementación anterior con websockets
         * event(new SpecialEvent($special, $channel));
        **/
    }

    public function createArray($booking_cta, $call_cta, $tel_cta, $tel_cta_es, $custom_cta, $name_cta, $link_cta, $name_cta_es, $link_cta_es) {


        $call_actions = [
            'bookingcta' => $booking_cta,
            'callcta' =>  
                $call_cta,
                [
                'tel' => $tel_cta,
                'tel_es' => $tel_cta_es
                ],
            'linkcta' =>  
                $custom_cta,
                [
                'namecta' => $name_cta,
                'link' => $link_cta,
                'namecta_es' => $name_cta_es,
                'link_es' => $link_cta_es
                ]
        ];

            return json_encode($call_actions, true);
    }
}
