<?php

namespace AvengersMG\MGCms2019\App\Cms\Components\Specials;

use Astrotomic\Translatable\Translatable;
use AvengersMG\MGCms2019\App\Cms\Components\Galleries\Interfaces\FeaturableInterface;
use AvengersMG\MGCms2019\App\Cms\Components\Specials\SpecialTranslation;
use AvengersMG\MGCms2019\App\Cms\Pages\Page;
use AvengersMG\MGCms2019\App\Cms\Sites\Site;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;

class Special extends Model implements FeaturableInterface
{
    use Translatable;

    protected $fillable =
        ['page_id', 'start_date', 'end_date', 'start_travel', 'end_travel', 'countdown', 'permanent', 'notification_time', 'cta'];

    protected $translatedAttributes = ['name', 'terms', 'notification_time', 'cta'];

    public function page()
    {
        return $this->belongsTo(Page::class);
    }

    /**
     * Método público, que sirve como accesor, para recuperar la
     * fecha de inicio.
     * @param  mixed    $value  Fecha a inyectar
     * @return string           Fecha en formato 'd-m-Y'
     */
    public function getStartDateAttribute($value)
    {
        return date('d-m-Y', strtotime($value));
    }

    /**
     * Método público, que sirve como accesor, para recuperar la
     * fecha de fin.
     * @param  mixed    $value  Fecha a inyectar
     * @return string           Fecha en formato 'd-m-Y'
     */
    public function getEndDateAttribute($value)
    {
        return date('d-m-Y', strtotime($value));
    }

    /**
     * Método estático público para recuperar consulta sobre las páginas
     * de todos los sitios que tienen fuente de imágenes establecida.
     *
     * @return Builder         Consulta filtrada
     */
    public static function ownerPagesQuery()
    {
        return Page::whereHas(
            static::getFeaturableRelationshipMethod()
        );
    }

    /**
     * Método público, que sirve como accesor, para recuperar la
     * fecha de inicio.
     *
     * @return string           Fecha formateada
     */
    public function getFeaturableStartDateAttribute()
    {
        /*
          2019-11-28: Usar la fecha de inicio regular en lugar de la de viaje
        */
        return $this->start_date;
    }

    /**
     * Método público, que sirve como accesor, para recuperar la
     * fecha de fin.
     *
     * @return string           Fecha formateada
     */
    public function getFeaturableEndDateAttribute()
    {
        /*
         2019-11-28: Usar la fecha de fin regular en lugar de la de viaje

         El valor de $this->attributes manda el crudo en Y-m-d
        */
        return $this->end_date;
    }

    /**
     * Método estático público para recuperar el formato en que las fechas
     * son retornadas por el objeto.
     *
     * @return string           Formato de fecha
     */
    public static function getFeaturableDateFormat()
    {
        return 'd-m-Y';
    }

    /**
     * Método público, que sirve como accesor, para recuperar el valor de
     * permanencia del objeto.
     *
     * @return boolean          Bandera de permanencia
     */
    public function getFeaturablePermanentAttribute()
    {
        return (boolean) $this->permanent;
    }

    /**
     * Método público estático para recuperar el nombre del método que describe
     * la relación que la fuente de imágenes tiene con la página.
     *
     * El resultado debe ser un método que está definido como relación en el
     * modelo Page.
     *
     * @return string Nombre del método
     */
    public static function getFeaturableRelationshipMethod()
    {
        return 'special';
    }

    public function getStartTravelAttribute($value)
    {
        return date(static::getFeaturableDateFormat(), strtotime($value));
    }

    public function getEndTravelAttribute($value)
    {
        return date(static::getFeaturableDateFormat(), strtotime($value));
    }

    public function getActiveCountdownAttribute()
    {
        return date('Y/m/d', strtotime($this->end_date));
    }

    public function getBookingWindowAttribute()
    {
        return "{$this->start_date} - {$this->end_date}";
    }

    public function getTravelWindowAttribute()
    {
        return "{$this->start_travel} - {$this->end_travel}";
    }

    /**
     * Funcion para activar el status de las promociones
     * que ya no estan activas.
     * Cambia el status a hidden
     */

    public function updatePageStatus()
    {
        if (!$this->permanent) {
            $this->page->update(['status' => 'hidden']);
        }
        $this->update(['active' => false]);
    }
}
