<?php

namespace AvengersMG\MGCms2019\App\Cms\Components\Galleries\Repositories;

use \Exception;
use AvengersMG\MGCms2019\App\Cms\BaseRepository\BaseRepository;
use AvengersMG\MGCms2019\App\Cms\Components\Galleries\Gallery;
use AvengersMG\MGCms2019\App\Cms\Components\Galleries\GalleryFeature;
use AvengersMG\MGCms2019\App\Cms\Components\Galleries\Item;
use AvengersMG\MGCms2019\App\Cms\Mediafiles\Mediafile;
use AvengersMG\MGCms2019\App\Cms\Pages\Page;
use Carbon\Carbon;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;

class GalleryRepository extends BaseRepository implements GalleryInterface
{
    public function __construct()
    {
        parent::__construct(Gallery::class);
    }

    public function create($data)
    {
        /** @var string Fecha y hora actual en formato para SQL */
        $now = Carbon::now()->toDateTimeString();

        /* Iniciar transacción */
        DB::beginTransaction();

        try {
            $gallery = call_user_func("{$this->model}::create", [
                'name' => $data['name'],
                'name_es' => $data['name_es'],
                'page_id' => $data['page_id']
            ]);

            if (Arr::has($data, 'images')) {
                /* Verificar que todos sean solo imágenes */
                $realImagesIds = Mediafile::imagesOnly()
                    ->whereIn('id', $data['images'])
                    ->pluck('id');

                /* Dejar solo los medios que sí son imágenes */
                if ($realImagesIds->count() != count($data['images'])) {
                    $data['images'] = Arr::where(function ($value) use ($realImagesIds) {
                        return $realImagesIds->contains($value);
                    });
                }

                /* Insertar si las imágenes siguen existiendo */
                if (count($data['images']) > 0) {
                    /** @var array[] Definiciones de ítems de galería */
                    $insertImageData = [];

                    /* Poblar definiciones */
                    foreach ($data['images'] as $priority =>  $image) {
                        $insertImageData[] = [
                            'gallery_id' => $gallery->id,
                            'mediafile_id' => $image,
                            'priority' => $priority,
                            'created_at' => $now
                        ];
                    }

                    /* Nuevos ítems */
                    Item::insert($insertImageData);
                }
            }

            /* Guardar cambios */
            DB::commit();
        } catch (Exception $e) {
            /* Deshacer cambios */
            DB::rollback();
        }

        return Page::find($data['page_id']);
    }

    /**
     * Método para actualizar el orden de los mediafiles asignados a la galería
     *
     * @param  mixed[] $data Lista de IDs de ítems en orden
     * @param  Gallery|int   $id   Galería o identificador de la galería
     * @return Gallery       Objeto modificado
     * @throws ModelNotFoundException|Exception
     */
    public function prepareToUpdate($data, $id)
    {
        if ($id instanceof Gallery) {
            $gallery = $id;
        } elseif (is_int($id)) {
            $gallery = $this->findOrFail($id);
        } else {
            throw new Exception("Dato no válido para galería");
        }

        /*
          2019-07-30: Refactorización de actualizador de orden (`priority`)
        */
        $nowForSqlField = Carbon::now()->toDateTimeString();

        $priorityChanges = collect($data);

        /* Iniciar transacción */
        DB::beginTransaction();

        try {
            /* Necesario establecer la prioridad como null antes
            de actualizar, para evitar restricciones de base de datos */
            $this->nullifyGalleryOrder($gallery);

            /* Necesario construir consulta cruda para evitar ciclo con múltiples llamadas y retornar las filas afectadas */
            Item::query()
                ->setBindings(
                    array_merge(
                        /* NOTA: Orden es imperativo */
                        $priorityChanges->mapWithKeys(function ($value, $index) {
                            return ["id{$value}" => $index];
                            /* $index = prioridad, $value = ID de ítem */
                        })->all(),
                        [
                            "isPriorityFixed" => 1,
                            /* Necesario establecer como número */
                            "updatedAt" => $nowForSqlField,
                            /* Necesario establecer
                            `updated_at` porque, si no,
                             el sistema lo implementa
                             usando el binding
                             por posiciones */
                        ]
                    )
                )
                ->update([
                    'priority' => DB::raw(
                        implode([
                            'CASE `id`',
                            $priorityChanges->map(function ($value, $index) {
                                return "WHEN {$value} THEN :id{$value}";
                                /* $index = prioridad, $value = ID de ítem */
                            })->implode(' '),
                            'ELSE `priority` END'
                        ], ' ')
                    ),
                    'is_priority_fixed' => DB::raw(':isPriorityFixed'),
                    'updated_at' => DB::raw(':updatedAt'),
                ]);

            /* Aplicar cambios */
            DB::commit();
        } catch (Exception $e) {
            /* Deshacer cambios */
            DB::rollback();
        }

        return $gallery->load('items');
    }

    public function updateOrInsert($request, $gallery)
    {
        $requestHasImages = $request->has('images');

        $requestHasGalleryFeatures = $request->has('gallery_features');

        $nowForSqlField = Carbon::now()->toDateTimeString();

        /*
            2019-07-31: Refactorización para evitar bucles y establecimiento
            de transacción por causa de múltiples consultas y operaciones
         */

        /* Iniciar transacción */
        DB::beginTransaction();

        try {
            /* IDs de mediafiles ya establecidos */
            $idsImagesAlreadyIn = $gallery->items()->pluck('mediafile_id');

            /* Clases de fuentes ya establecidos */
            $featuresAlreadyIn = $gallery->features()->pluck('featurable_type');

            /**
             * Arreglo de IDs de ítems a eliminar
             *
             * @var int[]
             */
            $items_to_delete = $idsImagesAlreadyIn->diff(($requestHasImages) ? $request->images : [])->all();

            /**
             * Arreglo de IDs de fuentes a eliminar
             *
             * @var int[]
             */
            $features_to_delete = $featuresAlreadyIn->diff(($requestHasGalleryFeatures) ? $request->gallery_features : [])->all();

            /* Establecer nombres de la galería y otros datos */
            $gallery->name = $request->name;
            $gallery->name_es = $request->name_es;
            $gallery->is_mobile = ($request->has('is_mobile')) ? $request->is_mobile : false;
            $gallery->save();

            /* Si fuentes de imágenes que no permanecerán, eliminarlas */
            if (!empty($features_to_delete)) {
                $this->removeFeatures($features_to_delete, $gallery->id);

                /* Restablecer fuentes de imágenes ya establecidas */
                $featuresAlreadyIn = $featuresAlreadyIn->diff($features_to_delete);
            }

            /* Si hay nuevas fuentes de imágenes para la galería, agregar */
            if ($requestHasGalleryFeatures) {
                /* Clases de fuentes de imágenes */
                $featuresCollection = collect($request->gallery_features);

                /* Clases de fuentes por establecer */
                $featuresToInsert = $featuresCollection->diff($featuresAlreadyIn);

                /* Crear las fuentes de la galería que no están establecidas */
                if ($featuresToInsert->isNotEmpty()) {
                    /* Necesario construir consulta cruda para evitar ciclo con múltiples llamadas */
                    GalleryFeature::insert(
                        $featuresToInsert->map(function ($value) use ($gallery, $nowForSqlField) {
                            return [
                                'gallery_id' => $gallery->id,
                                'featurable_type' => $value
                            ];
                        })->all()
                    );
                }
            }

            /* Si hay elementos que no permanecerán, eliminarlos */
            if (!empty($items_to_delete)) {
                $this->removeItems($items_to_delete, $gallery->id);

                /* Restablecer elementos ya establecidos */
                $idsImagesAlreadyIn = $idsImagesAlreadyIn->diff($items_to_delete);
            }

            /* Si hay imágenes solicitadas, añadir o actualizar */
            if ($requestHasImages) {
                /* IDs de mediafiles solicitadas, en orden */
                $imageCollection = collect($request->images);

                /* Verificar que todos sean solo imágenes */
                $realImagesIds = Mediafile::imagesOnly()
                    ->whereIn('id', $imageCollection->all())
                    ->pluck('id');

                /* Dejar solo los medios que sí son imágenes */
                if ($realImagesIds->count() != $imageCollection->count()) {
                    $imageCollection = $imageCollection->filter(function ($value) use ($realImagesIds) {
                        return $realImagesIds->contains($value);
                    });
                }

                /* Convertir valores (ID de mediafile) en llaves
                  y establecer índice de orden como nuevo valor */
                $imagesAlreadyIn = $idsImagesAlreadyIn->mapWithKeys(function ($value) use ($imageCollection) {
                    return ["{$value}" => $imageCollection->search($value)];
                });

                /* Convertir valores (ID de mediafile) en llaves
                  y establecer índice de orden como nuevo valor */
                $imagesToInsert = $imageCollection->diff($idsImagesAlreadyIn)->mapWithKeys(function ($value) use ($imageCollection) {
                    return ["{$value}" => $imageCollection->search($value)];
                });

                /* Actualizar los ítems de la galería que ya están establecidos */
                if ($imagesAlreadyIn->isNotEmpty()) {
                    /* Necesario establecer la prioridad como null antes
                    de actualizar, para evitar restricciones de base de datos */
                    $this->nullifyGalleryOrder($gallery);

                    /* Necesario construir consulta cruda para evitar ciclo con
                    múltiples llamadas y retornar las filas afectadas */
                    Item::query()
                        ->setBindings(
                            array_merge(
                                /* NOTA: Orden es imperativo */
                                $imagesAlreadyIn->mapWithKeys(function ($value, $index) {
                                    return ["id{$index}" => (int) $value];
                                    /* $index = ID de mediafile,
                                        $value = prioridad */
                                })->all(),
                                [
                                    "isPriorityFixed" => 1,
                                    /* Necesario establecer como número */
                                    "updatedAt" => $nowForSqlField,
                                    /* Necesario establecer
                                    `updated_at` porque, si no,
                                     el sistema lo implementa
                                     usando el binding
                                     por posiciones */
                                ]
                            )
                        )
                        ->whereRaw(
                            'gallery_id = :galleryId',
                            [
                                "galleryId" => $gallery->id,
                            ]
                        )
                        ->update([
                            'priority' => DB::raw(
                                implode([
                                    'CASE `mediafile_id`',
                                    $imagesAlreadyIn->map(function ($value, $index) {
                                        return "WHEN {$index} THEN :id{$index}";
                                        /* $index = ID de mediafile, $value = prioridad */
                                    })->implode(' '),
                                    'ELSE `priority` END'
                                ], ' ')
                            ),
                            'is_priority_fixed' => DB::raw(':isPriorityFixed'),
                            'updated_at' => DB::raw(':updatedAt'),
                        ]);
                }

                /* Crear los ítems de la galería que no están establecidos */
                if ($imagesToInsert->isNotEmpty()) {
                    /* Necesario construir consulta cruda para evitar ciclo con múltiples llamadas */
                    Item::insert(
                        $imagesToInsert->map(function ($value, $index) use ($gallery, $nowForSqlField) {
                            return [
                                'gallery_id' => $gallery->id,
                                'mediafile_id' => $index,
                                'priority' => $value,
                                'is_priority_fixed' => true,
                                /* Añadidos manualmente */
                                'created_at' => $nowForSqlField,
                                /* insert() no lo establece de
                                forma automática */
                            ];
                        })->all()
                    );
                }
            }

            /* Aplicar cambios */
            DB::commit();
        } catch (Exception $e) {
            /* Deshacer cambios */
            DB::rollback();
        }
    }

    public function getPage($page_id)
    {
        return Page::with(['mediafiles.type', 'mediafiles.translations'])->find($page_id);
    }

    /**
     * Método para actualizar el orden de las galerías asignadas a la página
     *
     * @param  mixed[]       $data Lista de IDs de galerías en orden
     * @param  Page|int      $id   Página o identificador de la página
     * @return boolean             Resultado de operación
     * @throws ModelNotFoundException|Exception
     */
    public function updatePageGalleriesOrder($data, $id)
    {
        if ($id instanceof Page) {
            $page = $id;
        } elseif (is_int($id)) {
            $page = Page::findOrFail($id);
        } else {
            throw new Exception("Dato no válido para página");
        }

        $retorno = false;

        $nowForSqlField = Carbon::now()->toDateTimeString();

        $priorityChanges = collect($data);

        /* Iniciar transacción */
        DB::beginTransaction();

        try {
            /* Necesario establecer la prioridad como null antes
            de actualizar, para evitar restricciones de base de datos */
            $this->nullifyPageGalleriesOrder($page);

            /* Necesario construir consulta cruda para evitar ciclo con múltiples llamadas y retornar las filas afectadas */
            $this->query()
                ->setBindings(
                    array_merge(
                        /* NOTA: Orden es imperativo */
                        $priorityChanges->mapWithKeys(function ($value, $index) {
                            return ["id{$value}" => $index];
                            /* $index = prioridad, $value = ID de ítem */
                        })->all(),
                        [
                            "updatedAt" => $nowForSqlField,
                            /* Necesario establecer
                            `updated_at` porque, si no,
                             el sistema lo implementa
                             usando el binding
                             por posiciones */
                        ]
                    )
                )
                ->update([
                    'priority' => DB::raw(
                        implode([
                            'CASE `id`',
                            $priorityChanges->map(function ($value, $index) {
                                return "WHEN {$value} THEN :id{$value}";
                                /* $index = prioridad, $value = ID de ítem */
                            })->implode(' '),
                            'ELSE `priority` END'
                        ], ' ')
                    ),
                    'updated_at' => DB::raw(':updatedAt'),
                ]);

            /* Aplicar cambios */
            DB::commit();

            /* Establecer dato de respuesta */
            $retorno = true;
        } catch (Exception $e) {
            /* Deshacer cambios */
            DB::rollback();
        }

        return $retorno;
    }

    /**
     * Método protegido para destruir el orden de los ítems de una galería
     * @param  Gallery $gallery Objeto de galería a usar
     * @return void
     */
    protected function nullifyGalleryOrder(Gallery $gallery)
    {
        Item::where('gallery_id', $gallery->id)->update([
            "priority" => null,
        ]);
    }

    /**
     * Método protegido para destruir el orden de las galerías de una página
     * @param  Page $page Objeto de página a usar
     * @return void
     */
    protected function nullifyPageGalleriesOrder(Page $page)
    {
        $this->query()->where('page_id', $page->id)->update([
            "priority" => null,
        ]);
    }


    /*
        2019-09-09: Cambio de visibilidad a protegido para evitar
        uso inapropiado pero permitir herencia
     */
    protected function removeItems($items_to_delete, $gallery_id)
    {
        /*
            2019-07-31: Refactorización para eliminar bucle de eliminación
         */
        Item::where('gallery_id', $gallery_id)->whereIn('mediafile_id', $items_to_delete)->delete();
    }

    /**
     * Método protegido para eliminar las fuentes de imágenes de una galería
     *
     * 2019-09-09: Cambio de visibilidad a protegido para evitar uso inapropiado
     * pero permitir herencia
     *
     * @param  string[] $gallery    Arreglo con nombres de clase
     * @param  int      $gallery_id Identificador de galería
     * @return void
     */
    protected function removeFeatures($features_to_delete, $gallery_id)
    {
        GalleryFeature::where('gallery_id', $gallery_id)->whereIn('featurable_type', $features_to_delete)->delete();
    }
}
