<?php

namespace AvengersMG\MGCms2019\App\Cms\BaseRepository;

use \App;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\QueryException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Storage;

abstract class BaseRepository implements BaseRepositoryInterface
{
    /** @var string Nombre de clase */
    protected $model;

    /** @var string Etiqueta de disco a utilizar. Protegida para heredar */
    protected $uploadDisk = 'MGCms2019';

    /**
     * Constructor de clase
     * @param string $className Nombre completo de clase
     */
    public function __construct($className)
    {
        $this->model = $className;
    }

    /**
     * Método mágico para redirigir todas las llamadas de métodos inexistentes al modelo
     *
     * @param  string $name       Nombre del método ejecutado
     * @param  array  $arguments  Argumentos enviados al método
     * @return mixed              Respuesta del método solicitado
     */
    public function __call($name, $arguments)
    {
        return call_user_func_array("{$this->model}::{$name}", $arguments);
    }

    public function list()
    {
        return call_user_func("{$this->model}::all");
    }

    public function update($params, $model = null)
    {
        if (!is_null($model)) {
            return $model->update($params->all());
        }

        /* Encontrar y actualizar específico */
        return call_user_func("{$this->model}::find", $params->id)->update($params->all());
    }

    public function delete($model = null)
    {
        if (!is_null($model)) {
            return $model->delete();
        }

        /* Borrar general */
        return call_user_func("{$this->model}::delete");
    }

    public function getMediafile()
    {
        return call_user_func("{$this->model}::mediafiles");
    }

    /**
     * Método público para subir archivos por medio de un Request
     *
     * @param  Request|array        $request    Objeto con información desde
     *                                          la página o arreglo con objeto
     *                                          en primer índice y nombre del
     *                                          campo de archivo en el segundo
     * @param  string               $uploadPath Directorio donde el archivo
     *                                          debe ser guardado
     * @param  boolean              $overwrite  Bandera de sobrescritura
     *                                          de archivo
     * @param  string|null          $uploadDisk Identificador de disco
     * @return Collection|string[]              Colección con etiquetas
     *                                          ("path" y "filename")
     */
    public function uploadFileByRequest($request, $uploadPath, $overwrite = false, $uploadDisk = null)
    {
        /* Nombre predeterminado de campo */
        $fieldName = 'file';

        /*
            Si es array, el primer índice es el objeto request, el segundo es
            el nombre del campo
        */
        if (is_array($request)) {
            $fieldName = $request[1];
            $request = $request[0];
        }

        /*
          Asegurar que la etiqueta de disco sea la predeterminada
          si es que no fue especificada
        */
        if (is_null($uploadDisk)) {
            $uploadDisk = $this->uploadDisk;
        }

        /* Obtener el nombre original para uso futuro */
        $originalFileName = $request->file($fieldName)->getClientOriginalName();

        /* Construir nuevo nombre tentativo, para uso futuro */
        $fileName = $this->slugifyFileName($originalFileName);

        /* Verificación de existencia de archivo con nombre tentativo. Si existe y no se solicita rescribir, cambiar nombre */
        if (!$overwrite) {
            /* Contador de incidencias */
            $counter = 0;

            /* Ciclar mientras el nombre de archivo esté ocupado */
            while (Storage::disk($uploadDisk)->exists($uploadPath . '/' . $fileName)) {
                /* Archivo ya existe. Sumar cuenta */
                $counter++;

                /* Crear nuevo nombre para guardar usando el contador */
                $fileName = $this->slugifyFileName($originalFileName, null, $counter);
            }
        }

        /* Guardar archivo en ubicación definitiva y retornar la dirección */
        return collect([
            'path' => $request->file($fieldName)->storeAs($uploadPath, $fileName, $uploadDisk),
            'filename' => $fileName
        ]);
    }

    /**
     * Método público para obtener un nombre
     * de archivo -con extensión- listo para uso en internet
     *
     * @param  string      $fileName     Nombre original de archivo
     * @param  string|null $separator    Cadena a utilizar para separar el slug.
     *                                   null usa el predeterminado '-'
     * @param  string      $append       Información adicional a escribir al final del nombre
     * @return string               Nombre de archivo filtrado a slug
     */
    public function slugifyFileName($fileName, $separator = null, $append = '')
    {
        /* Establecer separador predeterminado */
        if (is_null($separator)) {
            $separator = '-';
        }

        /* Posición del último punto */
        $lastPeriodPosition = mb_strrpos($fileName, '.');

        /* Nombre de archivo sin extensión. Si no hay una extensión definida, dejar mismo nombre */
        $fileNameNoExtension = (is_int($lastPeriodPosition))? mb_substr(
            $fileName,
            0,
            $lastPeriodPosition
        ) : $fileName;

        /* Crear nuevo nombre base. Para uso futuro */
        $newBaseFileName = str_slug($fileNameNoExtension, $separator);

        /* Construir nuevo nombre de archivo: Si append está vacío, no se toma en cuenta */
        $newFileName = (empty($append))? $newBaseFileName : ($newBaseFileName . $separator . str_slug($append, $separator));

        /* Retornar nuevo nombre */
        return str_replace(
            $fileNameNoExtension,
            $newFileName,
            $fileName
        );
    }
}
