<?php

namespace AvengersMG\MGCms2019\App\Cms\Availabilities\Repositories;

use \DateTime;
use \Exception;
use \PhpOffice\PhpSpreadsheet\IOFactory;
use \Throwable;
use AvengersMG\MGCms2019\App\Cms\Accommodations\Room;
use AvengersMG\MGCms2019\App\Cms\Availabilities\Availability;
use AvengersMG\MGCms2019\App\Cms\Availabilities\AvailabilitiesLogEntry;
use AvengersMG\MGCms2019\App\Cms\Availabilities\Exceptions\EmptyXLSFileAvailabilityException;
use AvengersMG\MGCms2019\App\Cms\Availabilities\Exceptions\InvalidXLSFileAvailabilityException;
use AvengersMG\MGCms2019\App\Cms\Availabilities\Repositories\AvailabilityInterface;
use AvengersMG\MGCms2019\App\Cms\BaseRepository\BaseRepository;
use AvengersMG\MGCms2019\App\Cms\Pages\Page;
use AvengersMG\MGCms2019\App\Cms\Sites\Site;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class AvailabilityRepository extends BaseRepository implements AvailabilityInterface
{
    private $months = [
        'enero' => '1',
        'febrero' => '2',
        'marzo' => '3',
        'abril' => '4',
        'mayo' => '5',
        'junio' => '6',
        'julio' => '7',
        'agosto' => '8',
        'septiembre' => '9',
        'octubre' => '10',
        'noviembre' => '11',
        'diciembre' => '12',
    ];

    public function __construct()
    {
        parent::__construct(Availability::class);
    }

    public function months($value)
    {
        return $this->months[$value];
    }

    /**
     * Función pública para recuperar las fechas donde la disponibilidad es nula para todo el hotel (todas las habitaciones están llenas) de acuerdo con un inicio y un fin. SI no hay fin, será hasta donde el sistema tenga registro.
     *
     * Si el formato no está especificado, el sistema asumirá 'Y-m-d' (YYYY-MM-DD)
     *
     * @param   date|null    $startDate    Fecha de inicio. Nulo si no la hay
     * @param   date|null    $endDate      Fecha de fin. Nulo si no la hay
     * @param   string|null  $outputFormat Formato compatible con date() en el que las fechas viene
     * @param   Site|null    $sitio        Sitio para el cual buscar. "null" para el actual
     * @throws  Exception|Throwable        Si el formato no es compatible con las fechas enviadas
     * @return  date[]                     Fechas de cierre en formato predeterminado o especificado
     *                                     por $outputFormat
     */
    public function getTotalBlackoutDates($startDate = null, $endDate = null, $outputFormat = null, $site = null)
    {
        /* Espacio para información de retorno */
        $retorno;

        /* Establecer el sitio */
        if (is_null($site)) {
            $site = Site::current();
        }

        /* Formato predeterminado */
        $defaultFormat = 'Y-m-d';

        /* Escribir fecha de hoy como predeterminada para fecha de inicio */
        if (is_null($startDate)) {
            $startDate = date(is_null($outputFormat)? $defaultFormat : $outputFormat, time());
        }

        try {
            /* Convertir las fechas al formato legible por SQL */
            if (!is_null($outputFormat)) {
                $startDate = DateTime::createFromFormat($outputFormat, $startDate)->format($defaultFormat);

                /* Convertir la fecha de fin si es que no es nula */
                if (!is_null($endDate)) {
                    $endDate = DateTime::createFromFormat($outputFormat, $endDate)->format($defaultFormat);
                }
            }

            /* Obtener las habitaciones */
            $roomIds = $this->getActiveRooms($site)->pluck('id');

            /* Arreglo de fechas en formato YYYY-MM-DD */
            $rawDates = DB::table('availabilities')
                            ->join('room_api_codes', 'room_api_codes.id', '=', 'availabilities.room_api_code_id')
                            ->select('availabilities.date')
                            ->where('availabilities.status', false)
                            ->where(function ($query) use ($startDate, $endDate) {
                                if (is_null($endDate)) {
                                    $query->whereDate('availabilities.date', '>=', $startDate);
                                } else {
                                    $query->whereBetween('availabilities.date', [$startDate, $endDate]);
                                }
                            })
                            ->whereIn('room_api_codes.room_id', $roomIds->toArray())
                            ->orderBy('availabilities.date')
                            ->groupBy('availabilities.date')
                            ->having(
                                DB::raw('count(*)'),
                                $roomIds->count()
                            )
                            ->pluck('availabilities.date')
                            ->all();

            /* Si hubo formato diferente, reformatear y enviar */
            if (!is_null($outputFormat)) {
                $retorno = collect($rawDates)->map(function ($date) use ($outputFormat, $defaultFormat) {
                    return DateTime::createFromFormat($defaultFormat, $date)->format($outputFormat);
                })->toArray();
            } else {
                $retorno = $rawDates;
            }
        } catch (Exception $e) {
            throw $e;
        } catch (Throwable $t) {
            throw $t;
        }

        /* Retornar resultados */
        return $retorno;
    }

    /**
     * Método público para crear nuevos registros de Availabitity de acuerdo con un archivo Xlsx subido
     *
     * @param  Request                  $request La petición enviada
     * @return void
     * @throws AvailabilityException    Errores diversos de importación
     */
    public function createAvailabilitiesFromXLSFileRequest(Request $request)
    {
        /** @var array[] Espacio para los meses que este repositorio debe crear */
        $monthsToCreate = [];

        /**
         * Espacio para arreglo de ID interna => ID de API (o ID interna => ID interna)
         * @var int[]|string[]
         */
        $roomsApiIds = [];

        /** @var string Tipo de archivo */
        $inputFileType = 'Xlsx';

        /** @var string Nombre original del archivo */
        $fileName = $request->file('file')->getClientOriginalName();

        /** @var string Tamaño original del archivo */
        $fileSize = $request->file('file')->getSize();

        /** @var string Tipo original del archivo */
        $fileContentType = $request->file('file')->getMimeType();

        /** @var string Nombre y dirección local de archivo subido (disco 'local' para que se guarde en "Storage")*/
        $pathFileName = $request->file('file')->storeAs('/uploads/excelfiles', $fileName, 'local');

        /** @var string Coordenadas (ColumnaFila) de la celda donde la lectura de rango empezará */
        $startCell = 'A1';

        /** @var boolean Bandera de nuevo formato (verdadero) o antiguo (falso) */
        $isNewFormat = false;

        /* Procesa el archivo para ser leído */
        $reader = IOFactory::createReader($inputFileType);
        $reader->setReadDataOnly(true);

        /* Leer contenido */
        $spreadsheet = $reader->load(storage_path('app/'.ltrim($pathFileName, '/')));

        /** @var string Coordenadas (ColumnaFila) de la celda donde la lectura de rango terminará */
        $endCell = $spreadsheet->getActiveSheet()->getHighestColumn().($spreadsheet->getActiveSheet()->getHighestRow());

        /** @var string Rango de celdas */
        $range = $startCell.':'.$endCell;

        /** @var (int,string)[] Arreglo unidimensional que representa la lectura del rango de celdas */
        $dataArray = $spreadsheet->getActiveSheet()->rangeToArray(
            $range, /* Rango de celdas a revisar */
            null, /*
              Valor con el cual las celdas vacías serán rellenadas

              2018-12-02: cambiado de 1 a null
            */
            true, /* ¿Calcular fórmulas?, equivalente a getCalculatedValue() por cada celda */
            true, /* ¿Formatear valores?, equivalente a getFormattedValue() por cada celda */
            true /* ¿Indexar arreglo por fila y columna de celda? */
        );

        /* Si la primera fila y columna es "-", es el formato antiguo, de lo contrario, es nuevo formato */
        $isNewFormat = ($dataArray['1']['A'] != '-');

        /** @var Collection|Room[] Habitaciones actuales */
        $activeRooms = $this->getActiveRooms();

        if ($isNewFormat) {
            /* mapWithKeys() explotará los códigos de API en lugar de retornar colección de colecciones */
            $roomsApiIdsCollection = $activeRooms->mapWithKeys(function ($room) {
                /*
                  Nuevo formato: Hacer arreglo de ID de API => ID interna de código de API
                  usando las habitaciones activas del sitio actual
                */
                return $room->apiCodes->mapWithKeys(function ($item, $key) {
                    return [
                        "{$item->api_code}" => $item->id
                    ];
                });
            });
        } else {
            $roomsApiIdsCollection = $activeRooms->mapWithKeys(function ($item, $key) {
                /*
                  Antiguo formato: Hacer arreglo de ID interna => ID interna de código de API
                  usando las habitaciones activas del sitio actual
                 */
                return [
                    "{$item->id}" => $item->apiCodes->first()->id,
                ];
            });
        }

        /** @var int[] Arreglo de IDs de habitación con códigos de API como llaves */
        $roomsApiIds = $roomsApiIdsCollection->toArray();

        /* Espacio para los pedazos */
        $chunks = [];

        /** @var boolean  Bandera que indica si el mes fue encontrado */
        $monthFound = false;

        /** @var boolean  Bandera que indica si el archivo tiene algún formato esperado */
        $hasExpectedFormat = false;

        /* Espacio de las filas por mes */
        $month = [];

        /* Mover el puntero al último índice del arreglo */
        end($dataArray);

        /* Obtener la etiqueta de la última fila */
        $lastRowKey = key($dataArray);

        /* Resetear arreglo para bucle */
        reset($dataArray);

        /** @var string Columna en la cual inician los días. 'C' es la conducta predeterminada */
        $firstDayColumn = 'C';

        /* Agrupar filas por mes */
        foreach ($dataArray as $rowKey =>$row) {
            /* Revisar si la fila es encabezado de mes */
            $isMonthHeaderRow = $this->isMonthHeaderRow($row);

            /* revisar si la fila es la última del archivo */
            $isLastRow = ($rowKey == $lastRowKey);

            /* La fila es considerada vacía si todas las celdas están llenas con el mismo contenido */
            $isRowEmpty = count(array_unique($row)) == 1;

            if (!$monthFound && $isMonthHeaderRow) {
                /* Mes encontrado. Añadir, cambiar bandera y pasar a la siguiente fila */
                $month[] = $row;
                $monthFound = true;

                if(!$hasExpectedFormat){
                    /* El formato ya fue encontrado, por lo que hay que establecer la bandera */
                    $hasExpectedFormat = true;
                }
            } elseif ($monthFound && ($isMonthHeaderRow || $isLastRow || $isRowEmpty) && !empty($month)) {
                /* Si es la última fila y es válida, añadir al mes antes de cerrar */
                if ($isLastRow && $this->isValidRow($row)) {
                    $month[] = $row;
                }

                /* Añadir el mes */
                $chunks[] = $month;

                /* Cerrar mes */
                $month = [];

                if ($isMonthHeaderRow) {
                    /* Si es un encabezado, iniciar un nuevo mes */
                    $month[] = $row;
                } else {
                    /* De lo contrario, dejar el encontrarlo para la siguiente iteración */
                    $monthFound = false;
                }
            } elseif ($monthFound && !$isMonthHeaderRow && $this->isValidRow($row)) {
                /* Sólo serán añadidas filas que sean válidas, aunque tengan disponibilidad */
                $month[] = $row;
            } elseif ($monthFound && $this->isDayHeaderRow($row)) {
                /*
                  Establecer la columna donde empiezan los números de días

                  ADVERTENCIA: Aunque esta sección rescribirá $firstDayColumn por
                  cada mes encontrado, no quiere decir que cada mes puede tener
                  su propio inicio de días. Al final, todo será procesado con el
                  último valor, por lo que es imperativo que todos los meses
                  inicien en la misma columna.
                 */
                $firstDayColumn = array_search(
                    /* 2019-12-02: Los datos numéricos YA NO vienen en float */
                    1,
                    $row,
                    true /* Búsqueda estricta: comparar tipo de dato */
                );
            }
        }

        /* Borrar variables innecesarias */
        unset($month, $monthFound);

        /* Verificar que cada mes sea válido ($month[0] fue verificado previamente) */
        foreach ($chunks as $k => $month) {
            $monthName = $this->getMonthFromRow($month[0]);
            $year = (int) $this->getYearFromRow($month[0]);

            if ($this->isYearAndMonthValid($year, $monthName)) {
                /* Válido: Añadir a los meses a crear */
                $monthsToCreate[] = $month;
            }
        }

        if ($hasExpectedFormat) {
            /* Sólo ejecutar si hay registros válidos */
            if (!empty($monthsToCreate)) {
                /* Escribir los nuevos registros */
                $this->prepareToCreateMultiple(
                    $monthsToCreate,
                    $roomsApiIds,
                    $firstDayColumn,
                    true /* Truncar la tabla antes de insertar */
                );

                /* Añadir un nuevo registro de actualización */
                AvailabilitiesLogEntry::create([
                    'user_id' => Auth::user()->id,
                    'file_name' => $fileName,
                    'file_size' => $fileSize,
                    'file_content_type' => $fileContentType,
                ]);
            } else {
                /* El archivo de disponibilidades es válido, pero está vacío. Arrojar la excepción */
                throw new EmptyXLSFileAvailabilityException("El archivo es válido pero no contiene datos de disponibilidad");
            }    
        } else {
            /* El archivo de disponibilidades no es válido. Arrojar la excepción */
            throw new InvalidXLSFileAvailabilityException("El archivo no tiene un formato esperado");
        }
    }

    /**
     * Método privado para verificar si la fila enviada es un encabezado de mes.
     * @param  mixed[]  $row             Fila de datos proveniente de un archivo de disponibilidad.
     * @return boolean                   El resultado
     */
    private function isMonthHeaderRow($row)
    {
        /* Un encabezado de mes debe tener un y sólo un nombre de mes y un y sólo un año*/
        return (count($this->getMonthsFromRow($row)) == 1) && (count($this->getYearsFromRow($row)) == 1);
    }

    /**
     * Método privado para retornar un arreglo con todas las coincidencias de nombre de mes.
     *
     * @param  mixed[]   $row Fila de datos proveniente de un archivo de disponibilidad.
     * @return string[]       Arreglo con etiquetas
     *                        (<nombre de fila> => <mes en minúsculas y sin acentos>)
     */
    private function getMonthsFromRow($row)
    {
        return array_intersect(
            array_map(
                function ($item) {
                    return trim(mb_strtolower($item));
                },
                $row
            ),
            array_keys($this->months)
        );
    }

    /**
     * Método privado para retornar la etiqueta de mes de un encabezado
     *
     * ADVERTENCIA: Este método no se asegura que la fila sea un encabezado válido.
     *
     * @param  mixed[] $row Fila de datos proveniente de un archivo de disponibilidad.
     * @return string       Etiqueta de columna
     */
    private function getMonthColumnTagFromRow($row)
    {
        /* Retornar la etiqueta de la primer incidencia de mes */
        return array_key_first(
            $this->getMonthsFromRow($row)
        );
    }

    /**
     * Método privado para retornar el nombre simplificado del mes de un encabezado
     *
     * ADVERTENCIA: Este método no se asegura que la fila sea un encabezado válido.
     *
     * @param  mixed[] $row Fila de datos proveniente de un archivo de disponibilidad.
     * @return string       Nombre simplificado de mes (minúsculas, sin acentos)
     */
    private function getMonthFromRow($row)
    {
        /* Retornar la primer incidencia de mes */
        return current(
            $this->getMonthsFromRow($row)
        );
    }

    /**
     * Método privado para retornar los años en un encabezado
     *
     * ADVERTENCIA: Este método no se asegura que la fila sea un encabezado válido.
     *
     * @return int[] Arreglo con etiquetas de los años en la fila
     */
    private function getYearsFromRow($row)
    {
        return array_filter(
            array_map(
                function ($item) {
                    /* 2019-12-02: Los datos numéricos YA NO vienen en float */
                    if (is_int($item)) {
                        return $item;
                    }

                    /* No pasaron los filtros */
                    return false;
                },
                $row
            ),
            function ($item) {
                /* Eliminar valores falsos */
                return (is_bool($item))? false : true;
            }
        );
    }

    /**
     * Método privado para retornar la etiqueta de año de un encabezado
     *
     * ADVERTENCIA: Este método no se asegura que la fila sea un encabezado válido.
     *
     * @param  mixed[] $row Fila de datos proveniente de un archivo de disponibilidad.
     * @return string       Etiqueta de columna
     */
    private function getYearColumnTagFromRow($row)
    {
        /* Retornar la etiqueta de la primer incidencia de mes */
        return array_key_first(
            $this->getYearsFromRow($row)
        );
    }

    /**
     * Método privado para retornar el año de un encabezado
     *
     * ADVERTENCIA: Este método no se asegura que la fila sea un encabezado válido.
     *
     * @param  mixed[] $row Fila de datos proveniente de un archivo de disponibilidad.
     * @return float        Año
     */
    private function getYearFromRow($row)
    {
        /* Retornar la primer incidencia de año */
        return current(
            $this->getYearsFromRow($row)
        );
    }

    /**
     * Método privado para verificar si la fila enviada es un encabezado de días para el mes.
     * @param  mixed[]  $row Fila de datos proveniente de un archivo de disponibilidad.
     * @return boolean       El resultado
     */
    private function isDayHeaderRow($row)
    {
        return (count(preg_grep('/^d(i|í|Í)a(s)?$/i', $row)) > 0);
    }

    /**
     * Método privado para determinar la validez del mes y año.
     *
     * Si el mes y el año son en el presente o futuro, es válido.
     *
     * @param  string  $year  Cadena con el año
     * @param  string  $month Cadena de nombre del mes, en minúsculas
     * @return boolean        Validez del mes y año
     */
    private function isYearAndMonthValid($year, $month)
    {
        /** @var int El año actual */
        $currentYear = (int) date('Y');

        /** @var int El mes actual */
        $currentMonth = (int) date('m');

        if ((((int) $year == $currentYear) && ((int) $this->months($month) >= $currentMonth)) || ((int) $year > $currentYear)) {
            return true;
        }

        /* Validación no pasó */
        return false;
    }

    /**
     * Inserción de información de disponibilidad por mes a base de datos.
     *
     * @param  array[]           $month               Arreglo de arreglos con información cruda proveniente
     *                                                del lector de XLS. Ver método
     *                                                createAvailabilitiesFromXLSFileRequest() para formato
     * @param  int[]|string[]    $roomsApiIds         Arreglo de ID de API de
     *                                                proveedor => ID interna de código de API o
     *                                                ID interna de habitación => ID interna de código de API
     * @param  string            $firstDayColumnTag   Etiqueta de la columna que contiene el primer día del mes.
     * @param  boolean           $truncate            Bandera para truncar la tabla
     *                                                antes de insertar. Pred.. falso
     * @return void
     */
    private function prepareToCreate($month, $roomsApiIds, $firstDayColumnTag = 'C', $truncate = false)
    {
        /* Espacio de información para inserción múltiple */
        $dataToInsert = $this->prepareMonthToInsert($month, $roomsApiIds, $firstDayColumnTag);

        /* Crear nuevos registros si existen en el espacio designado. Debe ser redirigido al modelo */
        if (!empty($dataToInsert)) {
            /* Truncar, si especificado */
            if($truncate){
                $this->truncate();
            }

            $this->insert($dataToInsert);
        }
    }

    /**
     * Implementación de la función prepareToCreate() para múltiples meses.
     *
     * @param  array[]           $months              Información cruda proveniente del lector de XLS.
     *                                                Ver método createAvailabilitiesFromXLSFileRequest()
     *                                                para formato
     * @param  int[]|string[]    $roomsApiIds         Arreglo de ID de API de
     *                                                proveedor => ID interna de código de API o
     *                                                ID interna de habitación => ID interna de código de API
     * @param  string            $firstDayColumnTag   Etiqueta de la columna que contiene el primer día del mes.
     * @param  boolean           $truncate            Bandera para truncar la tabla
     *                                                antes de insertar. Pred.. falso
     * @return void
     */
    private function prepareToCreateMultiple($months, $roomsApiIds, $firstDayColumnTag = 'C', $truncate = false)
    {
        /* Espacio de información para inserción múltiple */
        $dataToInsert = [];

        foreach ($months as $month) {
            /* Añadir los resultados al final de arreglo */
            $dataToInsert = array_merge($dataToInsert, $this->prepareMonthToInsert($month, $roomsApiIds, $firstDayColumnTag));
        }

        /* Crear nuevos registros si existen en el espacio designado. Debe ser redirigido al modelo */
        if (!empty($dataToInsert)) {
            /* Truncar, si especificado */
            if($truncate){
                $this->truncate();
            }

            $this->insert($dataToInsert);
        }
    }

    /**
     * Función privada para preparar la información recibida por mes para insertar.
     *
     * @param  array[]              $month              Información del mes.
     * @param  int[]|string[]       $roomsApiIds        Arreglo de ID de API de
     *                                                  proveedor => ID interna de código de API o
     *                                                  ID interna de habitación => ID interna de código de API
     * @param  string               $firstDayColumnTag  Etiqueta de la columna que contiene el primer día del mes.
     * @return array[]                                  La información preparada.
     */
    private function prepareMonthToInsert($month, $roomsApiIds, $firstDayColumnTag = 'C')
    {
        /* Información de retorno */
        $retorno = [];

        foreach ($month as $key => $row) {
            if ($key == 0) {
                /* La primer fila siempre es encabezado */
                $m = $this->months($this->getMonthFromRow($row));
                $y = (int) $this->getYearFromRow($row);
                $daysInMonth = cal_days_in_month(CAL_GREGORIAN, (int) $m, (int) $y);
            } elseif ($this->isValidRow($row)) {
                /*
                    Si la primer celda de la fila es un entero, es válida, pero
                    si no existe en las IDs de habitación, entonces saltar.
                */
                if (!array_key_exists((int) $row['A'], $roomsApiIds)) {
                    continue;
                }

                /* Valor de ID en columna 'A', (Legacy: Interna, Nuevo: proveedor) */
                $room_api_code_id = $roomsApiIds[(int) $row['A']];

                /* Contador de días */
                $day = 0;

                /* Bandera para indicar si ya fue encontrada la columna del primer día */
                $firstColumnFound = false;

                /* Procesar filas */
                foreach ($row as $columnTag => $column) {
                    if (!$firstColumnFound && ($columnTag == $firstDayColumnTag)) {
                        /* Primer columna encontrada. Cambiar bandera */
                        $firstColumnFound = true;
                    }

                    /* Procesar si ya fue encontrada */
                    if ($firstColumnFound) {
                        /* Columna a procesar */
                        $day++;

                        /* Desde la primera columna con día (especificada en $firstDayColumnTag) e iterar por la cantidad de días en el mes */
                        if ($day <= $daysInMonth) {
                            /*
                              Incluir para la inserción final sólo si la celda no está vacía (otra cosa que null indica "no disponible")

                              2019-12-02: Los datos numéricos YA NO vienen en float
                            */
                            if (!is_null($column)) {
                                $retorno[] = [
                                    'room_api_code_id' => $room_api_code_id,
                                    'status' => 0, /* Propiedad dejada por Legacy */
                                    'date' => Carbon::create($y, $m, $day),
                                    /* Añadidos timestamps porque Model::insert() no las actualiza */
                                    'created_at' => Carbon::now()->toDateTimeString(),
                                    'updated_at' => Carbon::now()->toDateTimeString()
                                ];
                            }
                        }
                    }
                }
            }
        }

        /* Regresar el resultado */
        return $retorno;
    }

    /**
     * Método privado para verificar que la fila enviada sea válida
     * @param  mixed[] $row Fila leída desde archivo. Ver createAvailabilitiesFromXLSFileRequest()
     *                      para formato
     * @return boolean      Respuesta de verificación
     */
    private function isValidRow($row)
    {
        /*
          2019-12-02: Los datos numéricos YA NO vienen en float

          Si la primer columna es numérica, es válida
        */
        return is_numeric($row['A']);
    }

    /**
     * Método privado para recuperar la lista de objetos Room activos en las páginas.
     *
     * Vienen de las páginas que tienen una habitación relacionada y que tengan al menos una traducción disponible
     *
     * @param Site|null          $site  Variable de sitio
     * @return Collection|Room[]        La lista de habitaciones activas
     */
    private function getActiveRooms($site = null)
    {
        /* Establecer el sitio */
        if (is_null($site)) {
            $site = Site::current();
        }

        /* Necesario llamar desde el modelo Page porque es posible que haya modelos Room abandonados asociados al mismo registro Page */
        return Page::bySite($site)
                    ->has('room.apiCodes')
                    ->whereHas('translations', function ($query) {
                            /*
                                2020-30-01: Deuda técnica de habitaciones repetidas resuelta con códigos de API
                                de venta independientes. Retorno a buscar solo páginas publicadas
                            */
                        $query->where('status', 'published');
                    })
                    ->with(['room.apiCodes', 'translations'])
                    ->get()
                    ->pluck('room');
    }
}