<?php

namespace AvengersMG\MGCms2019\App\Cms\Availabilities;

use AvengersMG\MGCms2019\App\Cms\Accommodations\RoomApiCode;
use Illuminate\Database\Eloquent\Model;

class Availability extends Model
{
    protected $table = "availabilities";
    protected $fillable = ['room_api_code_id', 'status', 'date'];

    /**
     * Crea la relación de uno a uno entre el modelo y el código de API
     * 
     * @return belongsTo La relación
     */
    public function apiCode()
    {
        return $this->belongsTo(RoomApiCode::class, 'room_api_code_id');
    }


    /**
     * Función pública -byRoomId()- que retorna los objetos con el cuarto indicado y valor "fecha" entre las fechas especificadas.
     *
     * Ésta es una implementación singular de $this->scopeByRoomIds()
     *
     * Nota: Los objetos son representaciones de una fecha, donde "status" indica si la fecha está disponible (false) o no disponible (true)
     *
     * @param Builder|Builder $query Consulta -inyectada- a modificar
     * @param int $roomId ID interna de la habitación
     * @param string $dateStart Cadena con formato de fecha que representa el arribo
     * @param string $dateEnd Cadena con formato de fecha que representa la salida
     */
    public function scopeByRoomId($query, $roomId, $dateStart, $dateEnd)
    {
        return $this->scopeByRoomIds($query, $roomId, $dateStart, $dateEnd);
    }

    /**
     * Función pública -byRoomIds()- que retorna los objetos con los cuartos indicados y valor "fecha" entre las fechas especificadas.
         *
         * Nota: Los objetos son representaciones de una fecha, donde "status" indica si la fecha está disponible (false) o no disponible (true)
     *
     * @param Builder|Builder $query Consulta -inyectada- a modificar
     * @param int[]|int $roomIds Entero o arreglo de IDs internas de las habitaciones
     * @param string $dateStart Cadena con formato de fecha que representa el arribo
     * @param string $dateEnd Cadena con formato de fecha que representa la salida
     */
    public function scopeByRoomIds($query, $roomIds, $dateStart, $dateEnd)
    {
        /* Si el ID es único (entero) convertir a arreglo */
        if (is_int($roomIds)) {
            $roomIds = [$roomIds];
        }

        /** @var date Fecha en formato YYYY-MM-DD */
        $dateStartYMD = date('Y-m-d', strtotime($dateStart));

        /** @var date Fecha en formato YYYY-MM-DD */
        $dateEndYMD = date("Y-m-d", strtotime($dateEnd));

        /* Retornar todas las disponibilidades de las habitaciones solicitadas */
        return $query->whereHas('apiCode', function ($query) use ($roomIds) {
            $query->whereIn('room_id', $roomIds);
        })->where('status', false)->whereBetween('date', [$dateStartYMD, $dateEndYMD])->orderBy('date', 'ASC');
    }
}
