<?php

namespace AvengersMG\MGCms2019\App\Cms\Apis\PriceTravel;

use \Exception;
use \stdClass;
use App;
use AvengersMG\MGCms2019\App\Cms\Apis\Interfaces\ApiInterface;
use Carbon\Carbon;
use GuzzleHttp\Client;
use Illuminate\Support\Collection;

class PriceTravelApi implements ApiInterface
{
    protected
        $arrivalDate, /** @var Carbon Fecha de llegada */
        $departureDate, /** @var Carbon Fecha de salida */
        $roomsData, /** @var array Información de petición de habitaciones */
        $remoteCachedData,/** @var mixed Información cruda recibida de la petición */
        $cachedData,/** @var Collection Información procesada recibida de la petición */
        $config, /** @var stdClass Objeto de configuración general de API */
        $lastException = null,  /** @var Exception|null Última excepción generada */
        $language = 'en',
        $currency = 'USD',
        $defaultDateFormat = 'Y-m-d';

    /**
     * Constructor de clase
     */
    public function __construct(){
        /* Establecer valores predeterminados de fechas y cuartos */
        $this->arrivalDate = Carbon::now()->addDays(2);
        $this->departureDate = $this->arrivalDate->addDays(2);
        $this->roomsData = [
            [
                'adults' => 2,
            ],
        ];

        /* Establecer el objeto de configuración */
        $this->config = new stdClass();
    }

    /**
     * Método público para establecer todas las credenciales que la API usa
     * @param array      $credentials Las credenciales
     * @return void
     */
    public function setCredentials(array $credentials) {
        /* Crear si no existe */
        if (!isset($this->config->credentials)) {
            $this->config->credentials = new stdClass();
        }

        /* Convertir arreglo a objeto para establecer credenciales */
        $this->config->credentials = json_decode(
            json_encode(
                $credentials
            ),
            FALSE
        );
    }

    /**
     * Método público para establecer la fecha de llegada
     * @param string      $arrivalDate Fecha de llegada
     * @param string|null $format      Formato en que la fecha está establecida
     * @return void
     */
    public function setArrivalDate(string $arrivalDate, $format = null) {
        /* Establecer formato en el que la fecha viene */
        if (is_null($format)) {
            $format = $this->defaultDateFormat;
        }

        /* Establecer la fecha de llegada */
        $this->arrivalDate = Carbon::createFromFormat($format, $arrivalDate);
    }

    /**
     * Método público para establecer la fecha de salida
     * @param string      $arrivalDate Fecha de salida
     * @param string|null $format      Formato en que la fecha está establecida
     * @return void
     */
    public function setDepartureDate(string $departureDate, $format = null) {
        /* Establecer formato en el que la fecha viene */
        if (is_null($format)) {
            $format = $this->defaultDateFormat;
        }

        /* Establecer la fecha de salida */
        $this->departureDate = Carbon::createFromFormat($format, $departureDate);
    }

    /**
     * Método público para recuperar la fecha de llegada
     * @param string|null $format      Formato en que la fecha está establecida
     * @return string
     */
    public function getArrivalDate($format = null) {
        /* Establecer formato en el que la fecha viene */
        if (is_null($format)) {
            $format = $this->defaultDateFormat;
        }

        /* Retornar en el formato deseado */
        return $this->arrivalDate->format($format);
    }

    /**
     * Método público para recuperar la fecha de salida
     * @param string|null $format      Formato en que la fecha está establecida
     * @return string
     */
    public function getDepartureDate($format = null) {
        /* Establecer formato en el que la fecha viene */
        if (is_null($format)) {
            $format = $this->defaultDateFormat;
        }

        /* Retornar en el formato deseado */
        return $this->departureDate->format($format);
    }

    /**
     * Método público para establecer el lenguaje
     * @param string      $language    Identificador de localización
     * @return void
     */
    public function setLanguage(string $language) {
        $this->language = $language;
    }

    /**
     * Método público para recuperar el lenguaje
     * @return string
     */
    public function getLanguage() {
        return $this->language;
    }

    /**
     * Método público para establecer la moneda
     * @param string      $currency    Identificador de moneda
     * @return void
     */
    public function setCurrency(string $currency) {
        $this->currency = $currency;
    }

    /**
     * Método público para recuperar la ocupación más grande solicitada
     * @return stdClass            Objeto con propiedades. (null = sin límite)
     *                                 stdClass(
     *                                     total => (int),
     *                                     adults => (int|null),
     *                                     children => (int|null)
     *                                 )
     */
    public function getLargestOccupancy() {
        /* Definir objeto de retorno */
        $retorno = new stdClass();
        $retorno->total = 0;
        $retorno->adults = null;
        $retorno->children = null;

        /* Buscar la habitación con la mayor ocupación solicitada */
        collect($this->roomsData)->each(function($room, $roomKey) use (&$retorno) {
            $parsedAdults = (int) $room['adults'];
            /* null = Sin límite de niños en el adaptador de API */
            $parsedChildren = isset($room['childAges']) ? count($room['childAges']) : null;

            $totalOccupancy = $parsedAdults + (is_null($parsedChildren) ? 0 : $parsedChildren);

            /* Si la ocupación de la habitación es mayor, tenerla como referencia */
            if ($totalOccupancy > $retorno->total) {
                $retorno->total = $totalOccupancy;
                $retorno->adults = $parsedAdults;
                $retorno->children = $parsedChildren;
            }
        });

        return $retorno;
    }

    /**
     * Método público para establecer la información de habitaciones
     * @param array      $roomsData    Arreglo con las definiciones de las
     *                                 habitaciones solicitadas en el
     *                                 siguiente formato ("childAges" es
     *                                 opcional en todos los cuartos):
     *                                      [
     *                                          [
     *                                              'adults' => (int),
     *                                              'childAges' => [
     *                                                  (int),
     *                                                  ...
     *                                              ],
     *                                          ],
     *                                          ...
     *                                      ]
     * @return void
     */
    public function setRoomsData(array $roomsData) {
        $this->roomsData = $roomsData;
    }

    /**
     * Método público para obtener la cuenta de habitaciones solicitadas
     * @return int Cantidad de habitaciones
     */
    public function getRoomsCount()
    {
        return (is_array($this->roomsData))
            ? count($this->roomsData)
            : 0;
    }

    /**
     * Método público para recuperar los datos crudos leídos del API, para evitar
     * una nueva llamada
     * @return mixed Los datos recuperados
     */
    public function getRemoteCachedData() {
        return $this->remoteCachedData;
    }

    /**
     * Método público para recuperar últimos resultados,
     * para evitar una nueva llamada
     * @return Collection Los datos recuperados
     */
    public function getCachedData() {
        if (empty($this->cachedData)) {
            return $this->getRemoteAvailabilityData();
        }

        /* No está vacío. Retornar caché */
        return $this->cachedData;
    }

    /**
     * Método público para retornar la información que el
     * adaptador retornará
     * @return Collection|stdClass[] Colección de objetos planos
     *                               con información normalizada de API:
     *                                  collect(
     *                                      stdClass(
     *                                         apiId => string,
     *                                         totalAmount => float,
     *                                         deepLink => string,
     *                                         ratePlans =>  collect(
     *                                            stdClass(
     *                                                totalAmount => float,
     *                                                averageRateWithTax => float,
     *                                                dailyRates => collect(
     *                                                    stdClass(
     *                                                        stayDate => Carbon,
     *                                                        totalAmount => float,
     *                                                        prePromotionalRate => float,
     *                                                        promotionMessage => float
     *                                                    ),
     *                                                    ...
     *                                                )
     *                                            ),
     *                                            ...
     *                                         )
     *                                      ),
     *                                      ...
     *                                  )
     *
     *
     */
    public function getRemoteAvailabilityData() {
        /* No hay caché. Recuperar datos */
        $retorno = collect();
        $client = new Client();

        /* Intentar recuperar información remota */
        try {
            $response = $client->get($this->getApiUrl(), [
                'auth' => $this->config->credentials->{$this->language},
                'verify' => (App::environment('local'))? false : true
                /* Sólo saltar verificación de SSL si el
                  ambiente es local */
            ]);

            /* Decodificar resultados crudos */
            $result = json_decode(
                $response->getBody()
            );

            /* Si no están regresados los planes o si no existen... */
            if (!isset($result->RatePlans) || empty($result->RatePlans)) {
                throw new Exception('El API no retornó planes');
            }

            /* Crear nueva colección con los planes */
            $ratePlans = collect($result->RatePlans);

            /* Establecer la información de cada habitación obtenida */
            $ratePlans->each(function($item, $key) use ($retorno){
                /* Establecer la clase a insertar con los nuevos valores */
                $currentRoom = new stdClass();
                $currentRoom->apiId = $item->RoomId;
                $currentRoom->totalAmount = $item->TotalAmount;
                $currentRoom->deepLink = $this->createDeepLink($item->RoomId, $item->RatePlanId);
                $currentRoom->ratePlans = collect();

                /* Iterar por los planes de cada habitación solicitada */
                collect($item->RoomRates)->each(function($item, $key) use ($currentRoom){
                    $ratePlan = new stdClass();

                    $ratePlan->totalAmount = $item->TotalAmount;
                    $ratePlan->averageRateWithTax = $item->AverageRateWithTax;
                    $ratePlan->dailyRates = collect();

                    collect($item->DailyRates)->each(function($item, $key) use ($ratePlan) {
                        $dailyRate = new stdClass();

                        /* Parece que  las fechas vienen en ISO 8601 */
                        $dailyRate->stayDate = Carbon::parse($item->StayDate);
                        $dailyRate->totalAmount = $item->TotalAmount;
                        $dailyRate->prePromotionalRate = $item->PrePromotionalRate;
                        $dailyRate->promotionMessage = $item->PromotionMessage;

                        /* Añadir a las tarifas por día */
                        $ratePlan->dailyRates->push($dailyRate);
                    });

                    /* Añadir habitación en plan de venta */
                    $currentRoom->ratePlans->push($ratePlan);
                });

                /* Añadir habitación a retorno */
                $retorno->push($currentRoom);
            });
        } catch (Exception $e) {
            /* Establecer excepción para revisión futura */
            $this->lastException = $e;
        }

        /* Destruir cliente HTTP */
        $client = null;

        /* Guardar la información para futuro acceso local, si es que existe */
        if (isset($result)) {
            $this->remoteCachedData = $result;
        }
        
        $this->cachedData = $retorno;

        /* Retornar colección */
        return $retorno;
    }

    /**
     * Método público para establecer toda la configuración que la API necesita
     *
     * @param array $configData Datos crudos a establecer
     * @return void
     */
    public function setConfig(array $configData) {
        /* Convertir arreglo a objeto para establecer credenciales */
        $this->config = json_decode(
            json_encode(
                $configData
            ),
            FALSE
        );
    }

    /**
     * Método de creación de URL para API de PriceTravel
     *
     * @return string La URL para urilizar en una petición HTTP GET
     */
    protected function getApiUrl(){
        $hotelId = $this->config->hotelId;

        $host = "https://api.pricetravel.com/services/hotels/{$hotelId}/availability";

        /* Estructura de variables para API */
        $uriParameters = [
            'currency' => $this->currency,
            'destinationId' => 140,
            'destinationType' => 6,
            'arrivalDate' => $this->arrivalDate->toDateString(), /* Y-m-d */
            'departureDate' => $this->departureDate->toDateString(),
            'language' => $this->language,
            'rooms' => $this->roomsData /* El arreglo ya viene estructurado desde el frontend */
        ];

        /* Armar URL completa para petición y retornar */
        return ($host.'?'.http_build_query($uriParameters));
    }

    /**
     * Método protegido para generar deep-links (vínculos que llevan al usuario a la página externa de reservación de PriceTravel )
     *
     * Los formatos de variables de la API de petición de precios de PriceTravel y de la página externa de reservaciones (también de PriceTravel) son radicalmente diferentes entre sí, tanto en nombres como en estructura. Ej.: El API usa toda la definición de cuartos en el arreglo "rooms", mientras que la página externa usa múltiples variables para cantidad (Rooms), adultos totales (Adults), niños totales (Kids), definición de ocupación (Room0.Adults, Room0.Kids -forzosa aunque sea 0-, Room1.Adults, Room1.Kids...), definición de edades (Room0.AgeKids=2,3 -Primer cuarto: un niño de 2 años y un niño de 3-). Aunque el sentido común indique que las variables parezcan redudantes e inconsistentes, son actualmente usadas por la página externa.
     *
     * @param  string|int $idRoom     ID externa (de PriceTravel) de la habitación
     * @param  string|int $RatePlanId ID externa (de PriceTravel) del plan de venta
     * @return string                 Vínculo completo de redirección a página externa
     */
    protected function createDeepLink($idRoom, $RatePlanId)
    {
        /* Este Endpoint sólo acepta este formato */
        $dateFormat = 'm/d/Y';

        /* Restructuración de variables */
        $uriParameters = [
            'idRoom' => $idRoom,
            'idRate' => $RatePlanId,
            'idHotel' => $this->config->hotelId,
            'Rooms' => count($this->roomsData),
            'CheckIn' => $this->arrivalDate->format($dateFormat),
            'CheckOut' => $this->departureDate->format($dateFormat),
            'Adults' => 0,
            'Kids' => 0
        ];

        /* Construir la URI utilizando OTRA API (que no está publicada) */
        collect($this->roomsData)->each(function($room, $indexRoom) use (&$uriParameters){
            /* Cantidad de niños */
            $kidsInRoom = (isset($room['childAges']))? count($room['childAges']) : 0;

            /* Poblar los totales */
            $uriParameters['Adults'] += (int) $room['adults'];
            $uriParameters['Kids'] += $kidsInRoom;

            /* Poblar la ocupación por habitación */
            $uriParameters["Room{$indexRoom}.Adults"] = (int) $room['adults'];
            $uriParameters["Room{$indexRoom}.Kids"] = $kidsInRoom;

            /* De ser necesario, poblar las edades de los niños en la habitación */
            if ($kidsInRoom > 0) {
                $uriParameters["Room{$indexRoom}.AgeKids"] = implode(',', $room['childAges']);
            }
        });

        /* Crear consulta compatible con dirección web y  */
        $roomsVars = http_build_query($uriParameters);

        /*
            Ejemplo de resultado:

            Suponiento una solicitud como esta:

                - Cuarto 1:
                    * Adultos: 2
                    * Niños: 2 (edades: 6 y 6)
                - Cuarto 2:
                    * Adultos: 2
                    * Niños: 2 (edades: 2 y 3)

            El resultado en $roomsVars es algo como esto:

            "idRoom=6036013&idRate=40965562&keyWordId=16&idHotel=324983&keyWordTable=Cities&Rooms=2&room0.Adults=2&room1.Adults=2&room0.AgeKids=6,6&room1.AgeKids=2,3&CheckIn=07/01/2019&CheckOut=07/03/2019&room0.Kids=2&room1.Kids=2"
        */

        /* Unirlo an endpoint final de la página externa de reservaciones y retornar */
        return $this->getHostDeepLink($roomsVars);
    }

    /**
     * Método protegido para poblar el valor del hostLink de acuerdo con el lenguaje
     * @param string $query Consulta URL a añadir
     * @return void
     */
    protected function getHostDeepLink($query)
    {
        /* Obtener desde configuraciones */
        $deepLinks = $this->config->hostDeepLinks;

        /* Establecer de acuerdo con el lenguage de la clase */
        $baseDeepLink = isset($deepLinks->{$this->language})
            ? $deepLinks->{$this->language}
            : $deepLinks->en;

        return empty($query)? $baseDeepLink : "{$baseDeepLink}?{$query}";
    }

    /**
     * Método público para recuperar la última excepción generada.
     * @return Exception
     */
    public function getLastException()
    {
        return $this->lastException;
    }

    /**
     * Método público para recuperar la cantidad total de adultos en la reservación
     * @return int
     */
    public function getTotalAdultCount()
    {
        /** @var int Cantidad de adultos */
        $retorno = 0;

        /* Solamente iterar si es que hay habitaciones */
        if ($this->getRoomsCount() > 0) {
            /* Iterar para sumar los adultos */
            collect($this->roomsData)->each(function($room) use (&$retorno) {
                $retorno += (int) $room['adults'];
            });
        }

        /* Retornar valor */
        return $retorno;
    }

    /**
     * Método público para recuperar la cantidad total de menores en la reservación
     * @return int
     */
    public function getTotalChildrenCount()
    {
        /** @var int Cantidad de menores */
        $retorno = 0;

        /* Solamente iterar si es que hay habitaciones */
        if ($this->getRoomsCount() > 0) {
            /* Iterar para sumar los menores */
            collect($this->roomsData)->each(function($room) use (&$retorno) {
                $retorno += isset($room['childAges']) ? count($room['childAges']) : 0;
            });
        }

        /* Retornar valor */
        return $retorno;
    }
}
