<?php

namespace AvengersMG\MGCms2019\App\Cms\Apis\Adapters;

use AvengersMG\MGCms2019\App\Cms\Apis\Calendar;
use AvengersMG\MGCms2019\App\Cms\Apis\Interfaces\ApiInterface;
use AvengersMG\MGCms2019\App\Cms\Accommodations\Rooms\RoomInterface;
use AvengersMG\MGCms2019\App\Cms\Availabilities\Availability;
use AvengersMG\MGCms2019\App\Cms\Availabilities\Repositories\AvailabilityInterface;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Carbon\Carbon;

class ApiAdapter
{
    protected
        /** @var ApiInterface Propiedad protegida que tiene las funciones de la API */
        $api,
        /** @var RoomInterface Repositorio de habitaciones */
        $roomRepository,
        /** @var AvailabilityInterface Repositorio de objetos de disponibilidad */
        $availabilityRepository;

    /**
     * Constructor de clase
     * @param ApiInterface          $api                   Clase con las
     *                                                     definiciones y
     *                                                     métodos a usar
     * @param AvailabilityInterface $availabilityInterface Repositorio de
     *                                                     objetos de
     *                                                     disponibilidad
     * @param RoomInterface         $roomInterface         Repositorio de
     *                                                     habitaciones
     */
    public function __construct(
        ApiInterface $api,
        AvailabilityInterface $availabilityInterface,
        RoomInterface $roomInterface
    )
    {
        /* Provisionar adaptador */
        $this->api = $api;
        $this->availabilityRepository = $availabilityInterface;
        $this->roomRepository = $roomInterface;
    }

    /**
     * Método mágico para redirigir todas las llamadas de métodos inexistentes al adaptado
     *
     * @param  string $name       Nombre del método ejecutado
     * @param  array  $arguments  Argumentos enviados al método
     * @return self               Retornar esta misma clase para daisy-chain
     */
    public function __call($name, $arguments)
    {
        /* Ejecutar el método en el API */
        call_user_func_array([$this->api, $name], $arguments);

        /* Retornar esta misma clase */
        return $this;
    }

    /**
     * Método para recuperar la propiedad protegida de API
     * @return ApiInterface El API al que este objeto está ligado
     */
    public function getApi()
    {
        return $this->api;
    }

    /**
     * Método público que recupera la disponibilidad remota de habitaciones
     * @return Illuminate\Support\Collection Disponibilidad remota normalizada
     */
    public function getAvailabilityData()
    {
        $largestOccupancy = $this->api->getLargestOccupancy();
        $apiLanguage = $this->api->getLanguage();

        /** @var Collection|Room[] Habitaciones con capacidad suficiente, traducción y códigos de API */
        $rooms = $this->getAndFilterLocalRoomsByFullCapacityWithLoadedRelations(
            $largestOccupancy->total,
            $apiLanguage
        );

        /**
         * Información del API
         * @var Illuminate\Support\Collection|stdClass[]
         */
        $result = $this->api->getRemoteAvailabilityData();

        /* Obtener todas las IDs de las habitaciones */
        $roomsIds = $rooms->map(function ($item, $key) {
            return $item->id;
        })->toArray();

        /**
         * Recuperar días de disponibilidad (o no) de las habitaciones obtenidas
         * entre la fecha de arribo y la fecha de salida ANTES del bucle, para evitar
         * inundación de consultas.
         *
         * @var Illuminate\Database\Eloquent\Collection|Availability[]
         */
        $availabilitiesDays = $this->availabilityRepository->byRoomIds(
            $roomsIds,
            $this->api->getArrivalDate(), /* Y-m-d */
            $this->api->getDepartureDate()
        )->get();

        /** @var Collection|Room[] Ciclar entre los cuartos obtenidos para comparar con planes obtenidos desde API */
        $roomsWithPlan = $this->filterAndOrderRoomsAndInjectPlansAndAvailabilities(
            $rooms,
            $largestOccupancy,
            $apiLanguage,
            $result,
            $availabilitiesDays
        );

        /* Retornar habitaciones con los planes encontrados */
        return $roomsWithPlan;
    }

    public function getAvailabilityDataForView()
    {
        /* Obtener habitaciones con información de disponibilidad */
        $roomsAvailable = $this->getAvailabilityData();

        /* Cargar relaciones adicionales */
        $roomsAvailable->loadMissing([
            'page.mediafiles.translations',
            'page.mediafiles.type'
        ]);

        /** @var int Total de cuartos seleccionados */
        $trooms = $this->api->getRoomsCount();

        /** @var int Diferencia de tiempo (en segundos) entre fechas */
        $datediff = strtotime($this->api->getDepartureDate()) - strtotime($this->api->getArrivalDate()); /* Y-m-d */

        /** @var int Cálculo de cantidad de días seleccionados */
        $total_days = (int) floor($datediff / (60 * 60 * 24));

        /* Crear objeto de funciones de días */
        $available = new Calendar(
            $this->api->getArrivalDate(),
            $this->api->getDepartureDate(),
            $this->api->getLanguage()
        );

        /* Necesario usar Carbon 2.0 o mayor */
        if ($this->api->getLanguage() == 'en') { // set time locale ingles
            $string_format = 'MMM. D, YYYY';
        } else {
            $string_format = 'D MMM, YYYY';
        }

        $dateformat = '<strong>' . Carbon::parse($this->api->getArrivalDate())->locale($this->api->getLanguage())->isoFormat($string_format) . '</strong>' . ' - ' . '<strong>' . Carbon::parse($this->api->getDepartureDate())->locale($this->api->getLanguage())->isoFormat($string_format) . '</strong>';

        $days['names'] = $available->createWeek();
        $days['nums'] = $available->createDays(new Availability);

        return [
            'arrivalDate' => $this->api->getArrivalDate(), /* Y-m-d */
            'departureDate' => $this->api->getDepartureDate(),
            'rooms' => $roomsAvailable,
            'total_days' => $total_days,
            'rooms_booked' => $trooms,
            'betweenDates' => $dateformat,
            'days' => $days
        ];
    }

    /**
     * Método protegido para recuperar las habitaciones
     * con capacidad suficiente, traducción y códigos de API.
     * 
     * @param  int                  Capacidad total de cuerpos
     * @param  string               Identificador de lenguaje (ej. "es", "en"...)
     * @return Collection|Room[]    Colección de resultados
     */
    protected function getAndFilterLocalRoomsByFullCapacityWithLoadedRelations($capacity, $language)
    {
        return $this->roomRepository->capacityRoom(
            $capacity,
            $language
        )->load(['translations' => function ($query) use ($language) {
            /* Cargar solamente la traducción correcta */
            $query->where('locale', $language);
        }, 'page.translations' => function ($query) use ($language) {
            /* Cargar solamente la traducción correcta */
            $query->where('locale', $language);
        }, 'apiCodes' => function($query) {
            $query->exists();
        }])->reject(function($room) use ($language) {
            /* Remover todas las habitaciones que estén escondidas */

            /*
                2020-01-16: Por deuda técnica, serán creadas habitaciones en las que el
                ID de API y el nombre es diferente, aunque físicamente la habitación sea
                la misma.

                2020-03-13: Resuelto con propuesta de códigos de API. Retornado a filtrar
                solamente con las que están publicadas.
            */
            return $room->page->translate($language)->status != "published";
        });
    }

    /**
     * Método protegido que inyecta los planes de venta o disponibilidades a las habitaciones.
     * Después, filtra por capacidad y disponibilidad para venta y ordena de acuerdo con el precio.
     * 
     * @param Collection|Room[]         $rooms              Colección de habitaciones
     *                                                      (preferible con las relaciones cargadas)
     * @param stdClass                  $capacity           Objeto que designa la ocupación máxima
     * @param string                    $language           Identificador de lenguaje (ej.: "es", "en"...)
     * @param Collection|stdClass[]     $apiResponseData    Respuesta de la API ya procesada
     * @param collection|Availability[] $availabilities     Disponibilidades de habitaciones
     * @return Collection|Room[]                            Colección filtrada y ordenada de habitaciones,
     *                                                      con planes o disponibilidades inyectados
     */
    protected function filterAndOrderRoomsAndInjectPlansAndAvailabilities($rooms, $capacity, $language, $apiResponseData, $availabilities)
    {   
        /** @var Collection|Room[] Habitaciones duplicadas con código de API establecido */
        $explodedRooms = $rooms
        ->map(function ($item, $key) {
            /*
                Clonar la habitación con cada código de API,
                para establecer la información de override
            */
            return $item->apiCodes->map(function ($apiCode) use ($item) {
                /* Clonar objeto, ya que el objeto pasa como referencia */
                $clonedItem = clone $item;

                /* Establecer el API en este código */
                $clonedItem->setMainApiCode($apiCode);

                /* Retornar copia modificada */
                return $clonedItem;
            });
        })
        ->flatten(1)
        ->map(function ($item, $key) use ($apiResponseData, $availabilities) {
            /* Ciclar planes obtenidos para buscar coincidencias */
            $room = $apiResponseData->where('apiId', $item->api_id)->first();

            /** @var Collection|int[] Códigos de disponibilidad de la habitación */
            $roomApiCodeIds = $item->apiCodes->pluck('id');

            /**
             * Recuperar días de disponibilidad (o no) de la habitación entre la
             * fecha de arribo y la fecha de salida.
             *
             * @var Illuminate\Support\Collection|Availability[]
             */
            $availabilityDays = $availabilities->filter(function ($itemAvail) use ($roomApiCodeIds) {
                return $roomApiCodeIds->contains($itemAvail->room_api_code_id);
            });

            /* Si hay alguna disponibilidad negativa, no añadir los planes ya que no está disponible aunque la API diga que sí */
            $isAvailable = count($availabilityDays) > 0 ? $this->verifyAvailability($availabilityDays) : true;

            /* Si está la habitación y es disponible, añadir información */
            if (!is_null($room) && $isAvailable) {
                $item->totalAmount = $room->totalAmount;
                $item->deepLink = $room->deepLink;
                $item->ratePlans = $room->ratePlans;
            } elseif ($isAvailable) {
                /* Retornar en el objeto la definición de disponibilidad por días de acuerdo con la base de datos local */
                $item->availability = $availabilityDays;
            }

            /* Retornar habitación rescrita */
            return $item;
        })
        ->reject(function ($item, $key) use ($capacity, $language) {
            /**
             * Si no existen planes ni disponibilidad, verdadero; si no, falso
             * @var boolean
             */
            $plansAndAvailabilityNotSet = (empty($item->ratePlans) && !isset($item->availability));

            $itemTranslation = $item->translate($language);

            /** @var boolean Capacidad de adultos rebasada  */
            $adultsNotMet = (is_null($capacity->adults))
                ? false
                : $capacity->adults > (int)($itemTranslation->adults);

            /** @var boolean Capacidad de menores rebasada  */
            $childrenNotMet = (is_null($capacity->children))
                ? false
                : $capacity->children > (int)($itemTranslation->childrens);

            /** @var boolean Capacidades rebasadas */
            $capacityNotMet = $adultsNotMet || $childrenNotMet;

            /*
                Quitar los que:
                    - No tengan ambos ratePlans y disponibilidad definida (no pueden ofrecerse) o...
                    - Tengan capacidad de adultos o menores más chica
            */
            return $plansAndAvailabilityNotSet || $capacityNotMet;
        })
        ->sortBy(function ($room, $key) {
            /* Ordenar las habitaciones de menor a mayor precio, enviando los que no tengan plan hasta el final */
            return (empty($room->totalAmount))? PHP_INT_MAX : $room->totalAmount;
        });

        /*
            flatten() retorna una colección regular. Es necesario convertirla
            en colección de Eloquent para cargar relaciones futuras.
        */
        return EloquentCollection::wrap($explodedRooms);
    }

    protected function verifyAvailability($availabilityDays){
        if(!empty($availabilityDays)){
            for ($i=1; $i <= count($availabilityDays); $i++) { 
                if (!empty($availabilityDays[$i]) && $availabilityDays[$i]->date != $this->api->getDepartureDate()) {
                    return false;
                }
            }
        }
        
        return true;
    }
}
