<?php

namespace AvengersMG\MGCms2019\App\Cms\Accommodations\Traits;

use \DateTime;
use Astrotomic\Translatable\Translatable;
use AvengersMG\MGCms2019\App\Cms\Accommodations\RoomTranslation;
use AvengersMG\MGCms2019\App\Cms\Accommodations\RoomApiCode;
use AvengersMG\MGCms2019\App\Cms\Mediafiles\FileType;

use Illuminate\Database\Eloquent\Model;

trait TranslatableHasRoomDataOverrides
{
    /** @var null|RoomApiCode Código de API a tomar como predeterminada */
    protected $mainApiCode = null;

    /* Crear alias para los métodos que vienen en el rasgo, para sobrescribirlos en este modelo */
    use Translatable {
        Translatable::attributesToArray as translatableTraitAttributesToArray;
        Translatable::fill as translatableTraitFill;
        Translatable::getAttribute as translatableTraitGetAttribute;
        Translatable::setAttribute as translatableTraitSetAttribute;
        Translatable::__isset as translatableTraitIsset;
    }

    public static function bootTranslatableHasRoomDataOverrides()
    {
        /* Booteando el original */
        static::bootTranslatable();

        static::saved(function (Model $model) {
            $model->saveRoomApiOverrides();
        });
    }

    /**
     * Método público para emular el primer código de API que tiene (compatibilidad hacia atrás)
     * @return string El código de API
     */
    public function getApiIdAttribute()
    {
        /** @var string|null Resultado de código de API  */
        $api_code = "";
        
        /* Hacer la sobrescritura si existe el código de API establecido */
        if ($this->issetMainApiCode()) {
            $api_code = $this->getMainApiCode()->api_code;
        } else {
            /** @var RoomApiCode|null Modelo de código */
            $api_code_model = $this->getMainApiCode();

            /* Escribir el primero, si existe */
            if (!is_null($api_code_model)) {
                $api_code = $api_code_model->api_code;
            }
        }

        /* Si es null, será convertido en cadena vacía */
        return strval($api_code);
    }

    /**
     * Método público para emular el primer modelo con tipo thumbnail que tiene (compatibilidad hacia atrás)
     * @return Mediafile|null Primer Mediafile considerado thumbnail
     */
    public function getThumbnailAttribute()
    {    
        /* Cargar la relación, si existe */
        $relation = $this->getMediafilesRelation();

        /* Hacer la sobrescritura si existe el código de API establecido */
        if ($this->issetMainApiCode()) {
            /** @var int|null Resultado de mediafile  */
            $mediafile_id = $this->getMainApiCode()->translateOrNew(
                $this->locale()
            )->thumbnail_mediafile_id;

            if (!is_null($mediafile_id)) {
                return $relation->find($mediafile_id);
            }
        }

        /* Retornar el primer considerado thumbnail */
        return $relation->where('type_id', FileType::getThumbnail()->id)->first();
    }

    /**
     * Sobrescritura de método en el modelo original y en el rasgo.
     * Basado en el método original de astrotomic/laravel-translatable
     *
     * Convert the model's attributes to an array.
     *
     * @return array
     */
    public function attributesToArray()
    {
        $attributes = $this->translatableTraitAttributesToArray();

        /* Hacer la sobrescritura si existe el código de API establecido */
        if ($this->issetMainApiCode()) {
            /** @var null|RoomApiCode Código de API actual */
            $apiCode = $this->getMainApiCode();

            $hiddenAttributes = $this->getHidden();

            /** @var RoomApiCodeTranslation Traducción del código de API */
            $apiCodeTranslation = $apiCode->translate(
                $this->locale()
            );

            foreach ($apiCodeTranslation->overrides as $override) {
                if (in_array($override->name, $hiddenAttributes)) {
                    continue;
                }

                $attributes[$override->name] = $override->data;
            }
        }  

        return $attributes;
    }

    /**
     * Sobrescritura de método en el modelo original y en el rasgo.
     * Basado en el método original de astrotomic/laravel-translatable
     *
     * Fill the model with an array of attributes.
     *
     * @param  array  $attributes
     * @return $this
     *
     * @throws \Illuminate\Database\Eloquent\MassAssignmentException
     */
    public function fill(array $attributes)
    {
        /* Hacer la sobrescritura si existe el código de API establecido */
        if ($this->issetMainApiCode()) {
            /** @var null|RoomApiCode Código de API actual */
            $apiCode = $this->getMainApiCode();

            /** @var RoomApiCodeTranslation Traducción del código de API */
            $apiCodeTranslation = $apiCode->translate(
                $this->locale()
            );

            foreach ($attributes as $key => $values) {
                $override = $apiCodeTranslation->overrides->firstWhere('name', $key);

                /* Si existe, sobrescribir y eliminar dato de arreglo */
                if (!is_null($override)) {
                    $override->data = $values;
                    unset($attributes[$key]);
                }
            }
        }
    
        /* De los atributos que quedan, escribir de acuerdo a rasgo */
        return $this->translatableTraitFill($attributes);
    }

    /**
     * Sobrescritura de método en el modelo original y en el rasgo.
     * Basado en el método original de astrotomic/laravel-translatable
     *
     * Get an attribute from the model.
     *
     * @param  string  $key
     * @return mixed
     */
    public function getAttribute($key)
    {
        /* Hacer la sobrescritura si existe el código de API establecido */
        if ($this->issetMainApiCode()) {
            /* Usar el método original de separación de atributo y localización */
            [$attribute, $locale] = $this->getAttributeAndLocale($key);

            /** @var null|RoomApiCode Código de API actual */
            $apiCode = $this->getMainApiCode();

            /** @var RoomApiCodeTranslation Traducción del código de API */
            $apiCodeTranslation = $apiCode->translate($locale);

            /** @var RoomDataOverride Dato a sobrescribir */
            $override = $apiCodeTranslation->overrides->firstWhere('name', $attribute);

            /* Override existe */
            if (!is_null($override)) {
                /*
                    Si existe mutador, modificar el valor en el arreglo interno
                    para poder aprovechar manipulación de tipos, mutadores
                    y campos de fecha
                */
                if ($this->hasGetMutator($attribute)) {
                    $this->attributes[$attribute] = $override->data;

                    return $this->getAttributeValue($attribute);
                }

                /* No hay mutador. Usar valores crudos */
                return $override->data;
            }
        }

        /* No hay código de API establecido. Retornar dato predeterminado */
        return $this->translatableTraitGetAttribute($key);
    }

    /**
     * Sobrescritura de método en el modelo original y en el rasgo.
     * Basado en el método original de astrotomic/laravel-translatable
     *
     * Set a given attribute on the model.
     *
     * @param  string  $key
     * @param  mixed  $value
     * @return mixed
     */
    public function setAttribute($key, $value)
    {
        /* Hacer la sobrescritura si existe el código de API establecido */
        if ($this->issetMainApiCode()) {
            /** @var null|RoomApiCode Código de API actual */
            $apiCode = $this->getMainApiCode();

            /** @var RoomApiCodeTranslation Traducción del código de API */
            $apiCodeTranslation = $apiCode->translate(
                $this->locale()
            );

            /** @var RoomDataOverride Dato a sobrescribir */
            $override = $apiCodeTranslation->overrides->firstWhere('name', $key);

            /*
                Override existe. Modificar el valor de la sobrescritura
                para poder aprovechar manipulación de tipos, mutadores
                y campos de fecha
            */
            if (!is_null($override)) {
                $override->data = $value;

                return $this;
            }
        }

        /* No hay código de API establecido. Retornar dato predeterminado */
        return $this->translatableTraitSetAttribute($key, $value);
    }

    /**
     * Sobrescritura de método en el modelo original y en el rasgo.
     * Basado en el método original de astrotomic/laravel-translatable
     *
     * Determine if an attribute or relation exists on the model.
     *
     * @param  string  $key
     * @return bool
     */
    public function __isset($key)
    {
        /** @var boolean Comprobación original */
        $originalIsset = $this->translatableTraitIsset($key);

        /* Hacer la sobrescritura si existe el código de API establecido */
        if ($this->issetMainApiCode()) {
            /** @var null|RoomApiCode Código de API actual */
            $apiCode = $this->getMainApiCode();

            /** @var RoomApiCodeTranslation Traducción del código de API */
            $apiCodeTranslation = $apiCode->translate(
                $this->locale()
            );

            /** @var boolean Espacio para revisar comprobación */
            $extraIsset = !is_null($apiCodeTranslation->overrides->firstWhere('name', $key));
            
            /* Retornar presencia */
            return $extraIsset || $originalIsset;
        }

        /* Retornar la conducta original */
        return $originalIsset;
    }

    /**
     * Método público para especificar qué código de API debe de usarse
     * al recuperar datos.
     * 
     * @param int|string|RoomApiCode|null Código de API a especificar.
     *                                      - int representa ID interna
     *                                      - string representa el código de API
     *                                      - RoomApiCode representa el modelo completo
     * @return void
     */
    public function setMainApiCode($apiCode)
    {
        /* Cargar la relación, si existe */
        $relation = $this->getApiCodesRelation();

        if (is_int($apiCode)) {
            /* Establecer por ID de objeto */
            $this->mainApiCode = $relation->firstWhere('id', $apiCode);
        } else if (is_string($apiCode)) {
            /* Establecer por código de API */
            $this->mainApiCode = $relation->firstWhere('api_code', $apiCode);
        } else if (is_a($apiCode, RoomApiCode::class)) {
            /* Establecer por objeto */
            $this->mainApiCode = $relation->firstWhere('id', $apiCode->id);
        } else if (is_null($apiCode)) {
            /* Remover */
            $this->mainApiCode = null;
        }
    }

    /**
     * Método público para recuperar el código de API debe de usarse
     * al recuperar datos. En caso que no esté especificado manualmente,
     * tomará la información del primero, si existe.
     * 
     * @return RoomApiCode|null
     */
    public function getMainApiCode()
    {
        /* Si ya existe, retornar el establecido */
        if ($this->issetMainApiCode()) {
            return $this->mainApiCode;
        }

        /* Cargar la relación, si existe */
        return $this->getApiCodesRelation()->first();
    }

    /**
     * Método público para verificar que el código de API
     * ha sido establecido.
     * 
     * @return boolean
     */
    public function issetMainApiCode()
    {
        /* Si ya existe, retornar el establecido */
        return !is_null($this->mainApiCode);
    }

    /**
     * Método protegido para cargar los códigos de API en caso de ser necesario
     * @return Collection|RoomApiCode[]
     */
    protected function getApiCodesRelation()
    {
        /*
            Si existen códigos de API, retornar el atributo del primero; si no, cadena vacía.

            Mientras la relación "apiCodes" sea cargada previamente al recuperar grupos,
            es posible evitar situación N + 1
        */
        if (!$this->relationLoaded('apiCodes')){
            $this->load('apiCodes.translations.overrides');
        }

        /* Cargar la relación, si existe */
        return $this->getRelation('apiCodes');
    }

    /**
     * Método protegido para cargar los mediafiles en caso de ser necesario
     * @return Collection|Mediafile[]
     */
    protected function getMediafilesRelation()
    {
        /*
            Si existen los mediafiles, retornar el atributo del primero; si no, cadena vacía.

            Mientras la relación "mediafiles" sea cargada previamente al recuperar grupos,
            es posible evitar situación N + 1
        */
        if (!$this->relationLoaded('mediafiles')){
            $this->load('mediafiles.translations');
        }

        /* Cargar la relación, si existe */
        return $this->getRelation('mediafiles');
    }

    /**
     *  Método protegido para guardar los modelos de sobrescritura cambiados
     *  al tiempo de guardar el modelo perteneciente.
     * 
     *  Basado en el método saveTranslations() de astrotomic/laravel-translatable
     * 
     *  @return boolean Resultado de guardado
     */
    protected function saveRoomApiOverrides()
    {
        $saved = true;

        /* Si la relación no está cargada o si no está establecido el código de API */
        if (!$this->relationLoaded('apiCodes') || !$this->issetMainApiCode()) {
            return $saved;
        } else if ($this->issetMainApiCode()) {
            /* Ciclar entre las sobrescrituras y guardar las que estén sucias */
            foreach ($this->getMainApiCode()->translate()->overrides as $override) {
                if ($saved && $override->isDirty()) {
                    if (!empty($connectionName = $this->getConnectionName())) {
                        $override->setConnection($connectionName);
                    }
    
                    /* Último resultado de actualización */
                    $saved = $override->save();
                }
            }
        }

        /* Retornar último resultado obtenido */
        return $saved;
    }
}