<?php

namespace AvengersMG\MGCms2019\App\Cms\Accommodations\Rooms;

use \Exception;
use AvengersMG\MGCms2019\App\Cms\Accommodations\Amenity;
use AvengersMG\MGCms2019\App\Cms\Accommodations\Categories\CategoryInterface;
use AvengersMG\MGCms2019\App\Cms\Accommodations\Room;
use AvengersMG\MGCms2019\App\Cms\Accommodations\RoomApiCode;
use AvengersMG\MGCms2019\App\Cms\BaseRepository\BaseRepository;
use AvengersMG\MGCms2019\App\Cms\MaxPax\MaxPax;
use AvengersMG\MGCms2019\App\Cms\Pages\Page;
use Carbon\Carbon;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Zend\Diactoros\Request;

class RoomRepository extends BaseRepository implements RoomInterface
{
    /* variables de estancia de clases protegidas */
    protected $category;
    /* constructor de clase */
    public function __construct(CategoryInterface $category)
    {
        parent::__construct(Room::class);
        $this->category = $category;
    }

    public function createGallery($gallery, $category_id, $is_update = [])
    {
        $arr_gallery = (!empty($is_update)) ? $is_update : [];

        foreach ($gallery as $value) {
            $name = $value->getClientOriginalName();
            $path = $value->storeAs('mediafiles/rooms/' . $category_id, $name, 'MGCms2019');

            array_push($arr_gallery, $path);
        }
        return $arr_gallery;
    }

    public function saveImage($image, $page_id)
    {
        $name = $image->getClientOriginalName();
        $name = explode('.', $name);
        $ext = $image->getClientOriginalExtension();
        $path = $image->storeAs('mediafiles/rooms/' . $page_id, Str::slug($name[0], '-') . '.' . $ext, 'MGCms2019');
        return $path;
    }

    /**
     * Método público para crear una habitación
     *
     * @param  Request $request Datos provenientes del controlador
     * @return Room|boolean     Habitación creada o false si hubo fallo
     */
    public function store($request)
    {
        /* cambia el idioma dependiendo de lo que contiene la variable $request->locale : en : es */
        app()->setLocale($request->locale);

        /* Iniciar transacción */
        DB::beginTransaction();

        try {
            $j_gallery = ($request->gallery) ?
                $this->createGallery($request->gallery, $request->category_id) : null;

            $floor_plan = ($request->floor_plan) ?
                $this->saveImage($request->floor_plan, $request->page_id) : null;

            /* Buscar la habitación de acuerdo con la página */
            $room = Room::where('page_id', $request->page_id)->first();

            /* Sólo crear nuevo objeto Room si no existe habitación en la página */
            if (is_null($room)) {
                $room = new Room();
            }

            /* Llama a los datos para guardar un nuevo registro de habitación */
            $room->category_id = $request->category_id;
            $room->page_id = $request->page_id;
            $room->floor_plan = $floor_plan;
            $room->gallery = json_encode($j_gallery);
            $room->map_3d = $request->mapa;
            $room->save();

            /** almacena datos para crear arreglo de capacidades */
            $capacidad = $request->capacity;
            $adultos = $request->adults;
            $ninos = $request->childrens;
            /** Contines los datos para la capacidad no permitida */
            $adultos_n = $request->adults_n;
            $ninos_n = $request->childrens_n;
            /**
             * almacena el resultado de la función local createArray
             * @var array
             */
            $capacidades = $this->createArray($capacidad, $adultos, $ninos, $adultos_n, $ninos_n);

            /* Si capacidad es diferente de un arreglo, arrojar excepción */
            if ($capacidades === false) {
                throw new Exception("No fue posible convertir las capacidades a un formato usable.");
            }
            /**
             * Convierte el arreglo en formaro Json
             * @var string
             */
            $arrayCapacity = json_encode($capacidades, true);

            /** @var Collection|string[] Configuraciones de idioma */
            $locales = collect(
                app()->make('translatable.locales')->all()
            );

            /*
                Crear registro en todos los idiomas habilitados mediante el foreach , es : en

                TODO: Revisar que esta parte esté ejecutándose correctamente con los traits booteables.
            */
            foreach ($locales->toArray() as $locale) {
                /* Traducción de cuarto, para mantener código DRY */
                $room_translation = $room->translateOrNew($locale);

                $room_translation->name = $request->name;
                $room_translation->description = $request->description;
                $room_translation->capacity = $request->capacity_room;
                $room_translation->adults = $request->adults_room;
                $room_translation->childrens = $request->childrens_room;
                $room_translation->beds = $request->beds;
                $room_translation->bathrooms = $request->bathrooms;
                $room_translation->floor_area = $request->floor_area;
                $room_translation->capacities = $arrayCapacity;
            }
            /*
                Guarda los datos del objeto mediante Eloquent

                También persiste los cambios hechos en $room_translation :O
            */
            $room->save();

            $room->amenities()->sync($request->amenity);

            /* Obtener y filtrar contra valores vacíos */
            $api_codes = collect($request->api_id)->unique()->filter();

            /* Si existen código de API, crear objetos */
            if ($api_codes->count() > 0) {
                $room->apiCodes()->createMany(
                    $api_codes->map(function ($api_code) use ($locales) {
                        $room_api_code_data = [
                            "api_code" => $api_code,
                        ];

                        /* Establecer las traducciones */
                        $locales->each(function ($locale) use (&$room_api_code_data) {
                            $room_api_code_data["{$locale}"] = [
                                "thumbnail_mediafile_id" => null,
                            ];
                        });

                        return $room_api_code_data;
                    })->toArray()
                );
            }

            /* Cambios realizados correctamente */
            DB::commit();
        } catch (Exception $e) {
            /* Deshacer cambios */
            DB::rollback();
            
            /*
                Error general

                TODO: Implementar excepciones específicas para mejorar el mantenimiento
            */
            $room = false;
        }

        return $room;
    }

    /**
     *
     * @param string[] $capacidad Capacidades máximas de ocupantes
     * @param string[] $adultos   Capacidades máximas de adultos
     * @param string[] $ninos     Capacidades máximas de niños
     * @return array[]|boolean    Retorna un arreglo con índices con etiquetas con capacidades
     *                            o false en caso de error
     */
    public function createArray($capacidad, $adultos, $ninos, $adultos_n, $ninos_n)
    {
        $capacidades = [];

        if (count(array_filter($capacidad)) == count(array_filter($adultos)) ) {

            $capacidades = array_map(function ($value, $i) use ($adultos, $ninos, $adultos_n, $ninos_n) {
              /* ADVERTENCIA: empty() toma "", "0", 0, 0.0, null, false y [] como vacío */
              if ((strlen((string) $value) > 0) && (strlen((string) $adultos[$i]) > 0) && (strlen((string) $ninos[$i]) > 0)) {
                    return [
                            "capacidad" => $value,
                            "yes" => [
                                "adultos" => $adultos[$i],
                                'ninos' => $ninos[$i]
                            ],
                            "no" => [
                                "adultos" => $adultos_n[$i],
                                'ninos' => $ninos_n[$i]
                            ]
                        //'capacidad' => $value, 'adultos' => $adultos[$i], 'ninos' => $ninos[$i]
                    ];
                }

                return false;
            }, $capacidad, array_keys($capacidad));

            return array_filter($capacidades);
        }

        return false;
    }

    /**
     * [prepareUpdate Método encargado de actualizar la habitación]
     * @param  [Request] $request [contiene los datos obtenidos desde la vista del formulario, método PUT]
     * @param  [objeto] $room    [contiene el modelo Room]
     * @return [Room|boolean]    [Retorna el objeto room o false si hubo error]
     */
    public function prepareUpdate($request, $room)
    {
        /** @var string Prefijo de ID de código de API */
        $prefix = RoomApiCode::fieldPrefix() . '_';

        /** @var string Identificador de la llave que contiene información para la traducción */
        $translationDataIdentifier = "translation";

        /** @var string Identificador de la llave que contiene información del alias del thumbnail */
        $thumbnailDataIdentifier = "thumbnail_mediafile_id";

        /** @var string Fecha para uso en SQL */
        $now_for_sql_field = Carbon::now()->toDateTimeString();

        /* Establecer localización */
        app()->setLocale($request->locale);

        /* Localización, para mantener código DRY */
        $locale = $request->locale;

        /* Iniciar transacción */
        DB::beginTransaction();

        try {
            $j_gallery = ($request->gallery) ?
            $this->createGallery($request->gallery, $request->category_id, $room->gallery) : null;

            $floor_plan = ($request->floor_plan) ?
                $this->saveImage($request->floor_plan, $room->page->id) : null;

            $room->amenities()->sync($request->amenity);

            /**
             * ROOM TABLE
             */
            $room->category_id = $request->category_id;
            $room->page_id = $request->page_id;
            $room->map_3d = $request->mapa;

            ($floor_plan) ? $room->floor_plan = $floor_plan : null;
            ($j_gallery) ? $room->gallery = json_encode($j_gallery) : null;

            /** almacena datos para crear arreglo de capacidades */
            $capacidad = $request->capacity;
            $adultos = $request->adults;
            $ninos = $request->childrens;
            /** Contines los datos para la capacidad no permitida */
            $adultos_n = $request->adults_n;
            $ninos_n = $request->childrens_n;
            /**
             * almacna el resultado de la función local createArray
             * @var array
             */
            $capacidades = $this->createArray($capacidad, $adultos, $ninos, $adultos_n, $ninos_n);

            /** Si capacidad es diferente de un arreglo, arrojar excepción */
            if ($capacidades === false) {
                throw new Exception("No fue posible convertir las capacidades a un formato usable.");
            }
            /**
             * Convierte el arreglo en formaro Json
             * @var string
             */
            $array_capacity = json_encode($capacidades, true);
            
            /**
             * ROOM TRANSLATION
             */
            $room_translation = $room->translateOrNew($locale);

            $room_translation->name = $request->name;
            $room_translation->description = $request->description;
            $room_translation->capacity = $request->capacity_room;
            $room_translation->adults = $request->adults_room;
            $room_translation->childrens = $request->childrens_room;
            $room_translation->beds = $request->beds;
            $room_translation->bathrooms = $request->bathrooms;
            $room_translation->floor_area = $request->floor_area;
            $room_translation->capacities = $array_capacity;

            /* Guardar modelo de traducción */
            $room_translation->save();

            /* Obtener y filtrar contra valores vacíos */
            $api_codes = collect($request->api_id)->unique()->filter();

            /* Obtener y filtrar contra valores vacíos */
            $overrides = collect($request->overrides)->mapWithKeys(function ($value, $tag) {
                return [
                    "{$tag}" => collect($value)->filter(function($data){
                        if (is_array($data) || is_object($data)) { 
                            return collect($data)->filter()->isNotEmpty();
                        }
                        return !empty($data);
                    })->toArray(),
                ];
            })->filter();

            /** @var string[] Códigos de API existentes */
            $existing_api_codes = $api_codes->filter(function ($value, $tag) use ($prefix) {
                return Str::startsWith($tag, $prefix);
            });

            /** @var string[] IDs de códigos de API existentes */
            $existing_api_codes_ids = $existing_api_codes->map(function ($value, $tag) use ($prefix) {
                return Str::after($tag, $prefix);
            });

            /**
             * Códigos de API nuevos
             * 
             * reject() mantendrá los índices originales
             * 
             * @var string[]
             */
            $new_api_codes = $api_codes->reject(function ($value, $tag) use ($prefix) {
                return Str::startsWith($tag, $prefix);
            });

            /* Actualizar códigos existentes */
            if ($existing_api_codes->isNotEmpty()) {
                /* Necesario construir consulta cruda para evitar ciclo con múltiples llamadas y retornar las filas afectadas */
                RoomApiCode::query()
                ->setBindings(
                    array_merge(
                        /* NOTA: Orden es imperativo */
                        $existing_api_codes_ids->mapWithKeys(function ($value) use ($existing_api_codes, $prefix) {
                            return ["id{$value}" => $existing_api_codes->get("{$prefix}{$value}")];
                        })->all(),
                        [
                            "updatedAt" => $now_for_sql_field,
                            /*
                                Necesario establecer `updated_at` porque, si no, el sistema lo implementa
                                usando el binding por posiciones
                            */
                        ]
                    )
                )
                ->update([
                    'api_code' => DB::raw(
                        implode([
                            'CASE `id`',
                            $existing_api_codes_ids->map(function ($value) {
                                return "WHEN {$value} THEN :id{$value}";
                            })->implode(' '),
                            'ELSE `api_code` END'
                        ], ' ')
                    ),
                    'updated_at' => DB::raw(':updatedAt'),
                ]);
            }

            /* Eliminar los códigos que no estén especificados */
            $room->apiCodes()->whereNotIn('id', $existing_api_codes_ids->toArray())->delete();
            
            /* Crear nuevos códigos de API */
            if ($new_api_codes->isNotEmpty()) {
                /**
                 * @var Collection|RoomApiCode[] Nuevos códigos de API creados
                 */
                $new_api_code_models = $room->apiCodes()->createMany(
                    $new_api_codes->map(function ($api_code, $api_code_tag) use ($overrides, $locale, $translationDataIdentifier, $thumbnailDataIdentifier) {
                        /** @var int|null ID de mediafile que sobrescribirá el thumbnail */
                        $thumbnail_id = null;

                        /* Sobrescribir el valor del thumbnail, si existe */
                        if ($overrides->has($api_code_tag)) {
                            $override_data = collect($overrides->get($api_code_tag));
                            if ($override_data->has($translationDataIdentifier)) {
                                $thumbnail_id = collect(
                                    $override_data->get($translationDataIdentifier)
                                )->get($thumbnailDataIdentifier);
                            }
                        }

                        return [
                            "api_code" => $api_code,
                            "{$locale}" => [
                                "{$thumbnailDataIdentifier}" => $thumbnail_id,
                            ],
                        ];
                    })->toArray()
                );
            }

            /** @var Collection|array[] Sobrescrituras existentes previas a esta ejecución */
            $overrides_preexisting = $overrides->filter(function ($value, $tag) use ($prefix) {
                return Str::startsWith($tag, $prefix);
            });

            /* Actualizar sobrescrituras de códigos preexistentes */

            /** @var Collection|RoomApiCode[] Traducciones de códigos preexistentes */
            $api_code_preexisting = $room->apiCodes()->with('translations.overrides')->whereIn('id', $existing_api_codes_ids->toArray())->get();

            /* Actualizar existentes, crear inexistentes y eliminar innecesarias por código */
            $api_code_preexisting->each(function ($value, $tag) use ($locale, $overrides_preexisting, $prefix, $translationDataIdentifier, $thumbnailDataIdentifier) {
                /** @var int|null ID de mediafile que sobrescribirá el thumbnail */
                $thumbnail_id = null;

                /** @var array Datos a sobrescribir */
                $override_data = $overrides_preexisting->get("{$prefix}{$value->id}", []);

                /* Sobrescribir el valor del thumbnail, si existe */
                if (array_key_exists($translationDataIdentifier, $override_data)) {
                    $thumbnail_id = collect(
                        $override_data[$translationDataIdentifier]
                    )->get($thumbnailDataIdentifier);
                }

                $translation = $value->translateOrNew($locale);

                /* Si la traducción no existe, guardarla */
                if (!$translation->exists) {
                    $translation->room_api_code_id = $value->id;
                }

                /* Establecer datos especiales para la traducción */
                $translation->thumbnail_mediafile_id = $thumbnail_id;
                $translation->save();

                /** @var Builder Consulta SQL filtrada */
                $translation_overrides_query = $translation->overrides();

                /* Si hay datos qué escribir, actualizar o crear */
                if (!empty($override_data)) {
                    foreach ($override_data as $name => $data) {
                        /* Saltar los datos enviados para la traducción */
                        if ($name == $translationDataIdentifier) {
                            continue;
                        }

                        /* Nota: Sentencia SQL en bucle. Disclaimer: Mantenible */
                        $translation_overrides_query->updateOrCreate(
                            ["name" => $name],
                            ["data" => $data]
                        );
                    }

                    /* Borrar las sobrescrituras no especificadas */
                    $translation_overrides_query->whereNotIn('name', array_keys($override_data))->delete();
                } else {
                    /* Eliminar todas las sobrescrituras */
                    $translation_overrides_query->delete();
                }
            });


            /* Crear nuevas sobrescrituras para códigos de API */
            if ($new_api_codes->isNotEmpty()) {
                /** @var Collection|array[] Sobrescrituras para códigos inexistentes previas a esta ejecución */
                $overrides_new = $overrides->reject(function ($value, $tag) use ($prefix) {
                    return Str::startsWith($tag, $prefix);
                });

                /* Crear nuevas sobrescrituras, si existen */
                if ($overrides_new->isNotEmpty()) {
                    $new_api_code_models->each(function ($value) use ($overrides_new, $locale, $new_api_codes) {
                        /* Si está en las sobrescrituras, usar */
                        if ($new_api_codes->contains($value->api_code)) {
                            /** @var Collection|array[] Datos de sobrescrituras a crear */
                            $overrides_to_create = collect(
                                $overrides_new->get(
                                    $new_api_codes->search($value->api_code)
                                )
                            )->map(function ($value, $tag) {
                                return [
                                    "name" => $tag,
                                    "data" => $value,
                                ];
                            });

                            /* Crear nuevas sobrescrituras */
                            $value->translate($locale)->overrides()->createMany($overrides_to_create->toArray());
                        }
                    });
                }
            }

            /* Guardar modelo principal */
            $room->save();

            /* Retornar modelo actualizado */
            $room->load(['apiCodes.translations.overrides', 'translations']);

             /* Cambios realizados correctamente */
             DB::commit();
        } catch (Exception $e) {
            /* Deshacer cambios */
            DB::rollback();

            /*
                Error general

                TODO: Implementar excepciones específicas para mejorar el mantenimiento
            */
            $room = false;
        }

        return $room;
    }
    /**
     * getCategories
     * @return [Response] [Retorna todas las amenidades existentes]
     */
    public function getCategories()
    {
        return $this->category->list();
    }
    /**
     * getAmenities
     * @return Response Retorna todas las amenidades existentes
     */
    public function getAmenities()
    {
        return Amenity::all();
    }

    /**
     * Función que retorna las habitaciones que tienen la capacidad suficiente y la traducción requerida
     *
     * @param int $totalPax Cantidad mínima de personas que la habitación debe permitir
     * @param string $language Lenguaje solicitado por la interfaz (Ej.: 'es', 'en'...)
     * @return Collection|Room[] Colección con las habitaciones encontradas
     */
    public function capacityRoom($totalPax, $language)
    {
        return call_user_func("{$this->model}::whereHas", 'translations', function ($query) use ($totalPax, $language) {
            $query->where('locale', $language)
                ->where('capacity', '>=', $totalPax);
        })->get();
    }

    /**
     * Método público para la eliminación de habitación y sus relaciones
     *
     * @param  Room     $room Habitación a eliminar
     * @return boolean        Resultado de eliminación
     */
    public function deleteAll($room)
    {
        $retorno = false;

        /* Iniciar transacción */
        DB::beginTransaction();

        try {
            /* Borrar traducciones y luego modelo */
            $room->deleteTranslations();
            $room->delete();

            /* Cambios realizados correctamente */
            DB::commit();

            /* Cambiar valor de retorno */
            $retorno = true;
        } catch (Exception $e) {
            /* Deshacer cambios */
            DB::rollback();
        }

        return $retorno;
    }
}
