<?php

namespace AvengersMG\MGCms2019\App\Cms\Accommodations;

use \DateTime;
use AvengersMG\MGCms2019\App\Cms\Accommodations\Category;
use AvengersMG\MGCms2019\App\Cms\Accommodations\RoomApiCode;
use AvengersMG\MGCms2019\App\Cms\Accommodations\Interfaces\ApiCodesRelationInterface;
use AvengersMG\MGCms2019\App\Cms\Accommodations\Traits\TranslatableHasRoomDataOverrides;
use AvengersMG\MGCms2019\App\Cms\Availabilities\Availability;
use AvengersMG\MGCms2019\App\Cms\MaxPax\MaxPax;
use AvengersMG\MGCms2019\App\Cms\Mediafiles\Mediafile;
use AvengersMG\MGCms2019\App\Cms\Pages\Page;
use Illuminate\Database\Eloquent\Model;

class Room extends Model implements ApiCodesRelationInterface
{
    /* Rasgo extendido de traducciones y códigos de API */
    use TranslatableHasRoomDataOverrides;

    protected $fillable = ['page_id', 'category_id', 'floor_plan', 'gallery', 'map_3d'];
    protected $appends = ['api_id', 'thumbnail'];
    public $translatedAttributes = ['locale', 'room_id', 'name', 'description', 'beds', 'bathrooms', 'capacity','adults', 'childrens', 'floor_area', 'capacities']; 
    
    public function page()
    {
        return $this->belongsTo(Page::class);
    }

    public function amenities()
    {
        return $this->belongsToMany(Amenity::class);
    }

    public function getGalleryAttribute($value)
    {
        return json_decode($value, true);
    }

    public function maxPax()
    {
        return $this->hasOne(MaxPax::class);
    }

    /**
     * Crea la relación de uno a muchos entre el modelo y las disponibilidades
     * @return hasManyThrough La relación
     */
    public function availabilities()
    {
        return $this->hasManyThrough(
            Availability::class,
            RoomApiCode::class,
            'room_id',
            'room_api_code_id',
            'id',
            'id'
        );
    }

    /**
     * Obtener una colección con las fechas no disponibles de la habitación
     *
     * @param   string|null         $outputFormat Formato compatible con date()
     *                                            en el que las fechas deben aparecer
     * @return  Collection|string[]               Colección de fechas
     * @throws  Exception|Throwable               Si el formato no es compatible
     *                                            con las fechas enviadas
     */
    public function getCurrentBlackoutDates($outputFormat = 'Y-m-d')
    {
        return $this->availabilities()->where('status', 0)->orderBy('date')->pluck('date')->map(function ($item) use ($outputFormat) {
            return (new DateTime($item))->format($outputFormat);
        });
    }

    /**
     * Crea la relación de uno a muchos entre el modelo y los códigos de API
     * 
     * @return hasMany La relación
     */
    public function apiCodes()
    {
        return $this->hasMany(RoomApiCode::class, 'room_id');
    }

    /**
     * Crea la relación de uno a muchos entre el modelo y mediafiles de la página padre
     * 
     * @return hasManyThrough La relación
     */
    public function mediafiles()
    {
        return $this->hasManyThrough(
            Mediafile::class,
            Page::class,
            'id',
            'mediafilable_id',
            'page_id',
        )->where('mediafilable_type', Page::class);
    }

    /**
     * Crea la relación de uno a muchos entre el modelo y las categorías
     * 
     * @return belongsTo La relación
     */
    public function category()
    {
        return $this->belongsTo(Category::class, 'category_id');
    }
}
